package basehttpserver

import (
	"context"
	"io"
	"log"
	"net"
	"net/http"
	"time"
)

// Config controls your server.  You can usually leave this blank
type Config struct {
	// ListenAddr is the address to listen on
	ListenAddr string
	// ShutdownWait is the timeout for graceful shutdown
	ShutdownWait time.Duration
	// Which HTTP endpoint to serve health checks on
	HealthcheckEndpoint string
}

func (c *Config) listenAddr() string {
	if c == nil || c.ListenAddr == "" {
		return ":8080"
	}
	return c.ListenAddr
}

func (c *Config) shutdownWait() time.Duration {
	if c == nil || c.ShutdownWait == 0 {
		return time.Second * 5
	}
	return c.ShutdownWait
}

func (c *Config) healthcheckEndpoint() string {
	if c == nil || c.HealthcheckEndpoint == "" {
		return "/debug/running"
	}
	return c.HealthcheckEndpoint
}

// Logger can log key/value strings
type Logger interface {
	Log(keyvals ...interface{})
}

// HTTPServer is the minimal needed parts of a HTTP server that supports listener interception, health checks, and graceful
// exits, etc
type HTTPServer struct {
	// Config is optional configuration
	Config *Config `nilcheck:"ignore"`
	// Mux is the HTTP handler. Routes can be added to it
	Mux http.ServeMux
	// CreateListener is an optional net.Listener factory. This may be useful for adding proxy protocol support. Defaults to net.Listen
	CreateListener func(network string, address string) (net.Listener, error)
	// Log is an optional logger. Defaults to not logging
	Log Logger
	// Server is the HTTP server
	Server http.Server
	// OnListen is an optional callback for getting the listener address
	OnListen func(net.Addr) `nilcheck:"ignore"`
	listener net.Listener
}

// Setup registers all the HTTPHandlers on the server's mux
func (h *HTTPServer) Setup() error {
	h.Mux.Handle(h.Config.healthcheckEndpoint(), http.HandlerFunc(h.Health))
	var err error
	if h.CreateListener != nil {
		h.listener, err = h.CreateListener("tcp", h.Config.listenAddr())
	} else {
		h.listener, err = net.Listen("tcp", h.Config.listenAddr())
	}
	if err == nil && h.OnListen != nil {
		h.OnListen(h.listener.Addr())
	}
	return err
}

// Health allows us to health check that the server is up and running.  It should return 'OK'
func (h *HTTPServer) Health(rw http.ResponseWriter, req *http.Request) {
	if _, err := io.WriteString(rw, "OK"); err != nil {
		h.log("err", err)
	}
}

// Start begins serving traffic on the server
func (h *HTTPServer) Start() error {
	h.Server.Handler = &h.Mux
	if h.Server.ErrorLog == nil && h.Log != nil {
		h.Server.ErrorLog = log.New(&ioToLog{Log: h.Log}, "", 0)
	}
	h.log("addr", h.listener.Addr().String(), "starting listen on server")
	err := h.Server.Serve(h.listener)
	if err != nil && err != http.ErrServerClosed {
		h.log("err", err, "server finished listening unexpectedly")
		return err
	}
	return nil
}

// Close shutsdown the server
func (h *HTTPServer) Close() error {
	shutdownCtx, cancel := context.WithTimeout(context.Background(), h.Config.shutdownWait())
	defer cancel()
	return h.Server.Shutdown(shutdownCtx)
}

func (h *HTTPServer) log(kvs ...interface{}) {
	if h.Log != nil {
		h.Log.Log(kvs...)
	}
}

type ioToLog struct {
	Log Logger
}

var _ io.Writer = &ioToLog{}

func (i *ioToLog) Write(p []byte) (n int, err error) {
	i.Log.Log(string(p))
	return len(p), nil
}
