package ctxtrace

import (
	"context"
	"encoding/json"
	"sync"
)

type logTypes int

const (
	traceKey logTypes = iota
)

// Trace lives along side a context and allows us to add arbitrary key/value pairs to the context so we can
// later log them out inside the access logs
type Trace struct {
	vals map[string]interface{}
	mu      sync.Mutex
}

func (t *Trace) init() {
	if t.vals == nil {
		t.vals = make(map[string]interface{})
	}
}

var _ json.Marshaler = &Trace{}

// Vals returns a copy of the values tracked by this trace
func (t *Trace) Vals() (map[string]interface{}) {
	if t == nil {
		return nil
	}
	t.mu.Lock()
	defer t.mu.Unlock()

	t.init()
	ret := make(map[string]interface{}, len(t.vals))
	for k, v := range t.vals {
		ret[k] = v
	}
	return ret
}

// MarshalJSON encodes the trace by only considering the internal values
func (t *Trace) MarshalJSON() ([]byte, error) {
	if t == nil {
		return []byte("{}"), nil
	}
	t.mu.Lock()
	defer t.mu.Unlock()

	t.init()
	return json.Marshal(t.vals)
}

// Set a named key to a value
func (t *Trace) Set(key string, value interface{}) {
	if t == nil {
		return
	}
	t.mu.Lock()
	defer t.mu.Unlock()
	t.init()
	t.vals[key] = value
}

// Inc a named key by a value
func (t *Trace) Inc(key string, value float64) {
	if t == nil {
		return
	}
	t.mu.Lock()
	defer t.mu.Unlock()
	t.init()
	v, exists := t.vals[key]
	if !exists {
		t.vals[key] = value
		return
	}
	asFloat, isFloat := v.(float64)
	if isFloat {
		t.vals[key] = asFloat + value
	}
}


// WithTrace initialises a context with a traceBody so we can inject key/value pairs into the context later
func WithTrace(ctx context.Context) context.Context {
	if existingTrace := GetTrace(ctx); existingTrace != nil {
		return ctx
	}
	return context.WithValue(ctx, traceKey, &Trace{})
}

// GetTrace extracts the trace created with `WithTrace`, so we can add key/value pairs to it
func GetTrace(ctx context.Context) *Trace {
	retI := ctx.Value(traceKey)
	if retI == nil {
		return nil
	}
	return retI.(*Trace)
}

