package ctxtrace_test

import (
	"code.justin.tv/hygienic/ctxtrace"
	"context"
	"encoding/json"
	"github.com/stretchr/testify/assert"
	"testing"
)

func TestWithTrace(t *testing.T) {
	t.Run("created", func(t *testing.T) {
		ctx := context.Background()
		ctx2 := ctxtrace.WithTrace(ctx)
		assert.NotEqual(t, ctx, ctx2)
	})
	t.Run("repeat", func(t *testing.T) {
		ctx := context.Background()
		ctx2 := ctxtrace.WithTrace(ctx)
		ctx3 := ctxtrace.WithTrace(ctx2)
		assert.Exactly(t, ctx2, ctx3)
	})
}

func TestGetTrace(t *testing.T) {
	t.Run("unset", func(t *testing.T) {
		ctx := context.Background()
		assert.Nil(t, ctxtrace.GetTrace(ctx))
	})
	t.Run("setonce", func(t *testing.T) {
		ctx := context.Background()
		ctx2 := ctxtrace.WithTrace(ctx)
		assert.Nil(t, ctxtrace.GetTrace(ctx))
		assert.NotNil(t, ctxtrace.GetTrace(ctx2))
	})
}

func TestTrace_Set(t *testing.T) {
	t.Run("string", func(t *testing.T) {
		ctx := context.Background()
		ctx = ctxtrace.WithTrace(ctx)
		ctxtrace.GetTrace(ctx).Set("hi", "bob")
		assert.Equal(t, ctxtrace.GetTrace(ctx).Vals(), map[string]interface{} {
			"hi": "bob",
		})
	})
}

func TestTrace_Inc(t *testing.T) {
	t.Run("int", func(t *testing.T) {
		ctx := context.Background()
		ctx = ctxtrace.WithTrace(ctx)
		ctxtrace.GetTrace(ctx).Inc("count", 1)
		assert.Equal(t, ctxtrace.GetTrace(ctx).Vals()["count"], float64(1))
	})
	t.Run("float64", func(t *testing.T) {
		ctx := context.Background()
		ctx = ctxtrace.WithTrace(ctx)
		ctxtrace.GetTrace(ctx).Inc("count", 1)
		ctxtrace.GetTrace(ctx).Inc("count", 2.0)
		assert.Equal(t, ctxtrace.GetTrace(ctx).Vals()["count"], float64(3))
	})
}

func TestTrace_Vals(t *testing.T) {
	t.Run("string", func(t *testing.T) {
		ctx := context.Background()
		ctx = ctxtrace.WithTrace(ctx)
		ctxtrace.GetTrace(ctx).Set("count", 1)
		assert.Equal(t, ctxtrace.GetTrace(ctx).Vals()["count"], 1)
	})
}

func TestTrace_MarshalJSON(t *testing.T) {
	type example struct {
		Name string `json:"name"`
	}
	type expectedVals struct {
		Name string `json:"name"`
		Count int `json:"count"`
		Place example
	}
	t.Run("mixed", func(t *testing.T) {
		ctx := context.Background()
		ctx = ctxtrace.WithTrace(ctx)
		ctxtrace.GetTrace(ctx).Set("name", "bob")
		ctxtrace.GetTrace(ctx).Inc("count", 1)
		ctxtrace.GetTrace(ctx).Set("place", &example{
			Name: "home",
		})
		x, err := json.Marshal(ctxtrace.GetTrace(ctx))
		assert.Nil(t, err)
		var into  expectedVals
		assert.Nil(t, json.Unmarshal(x, &into))
		assert.Equal(t, "bob", into.Name)
		assert.Equal(t, 1, into.Count)
		assert.Equal(t, example{
			Name: "home",
		}, into.Place)
	})
}

const (
	ctxKeyBody = iota
)

type ContextTrace struct {
	Body string
}

type TraceWrap struct {
	*ContextTrace
}

func WithTrace(ctx context.Context) context.Context {
	return context.WithValue(ctx, ctxKeyBody, &ContextTrace{})
}

func GetTrace(ctx context.Context) *ContextTrace {
	ret, exists := ctx.Value(ctxKeyBody).(*ContextTrace)
	if exists {
		return ret
	}
	// Return a dummy object so users don't have to worry about nil panics
	return &ContextTrace{}
}


func TestA(t *testing.T) {
	ctx := context.Background()

}
