package dcache

import (
	"encoding/json"
	"io/ioutil"
	"os"
	"sync"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
)

// Cached allows caching distconf variables
type Cached struct {
	PopulateFrom distconf.Distconf
	CacheFile    string
	CacheSection string
	Log          log.Logger

	loadedFile modifyableFormat
	loaded     sync.Once
}

type modifyableFormat struct {
	fileFormat
	mu sync.Mutex
}

type fileFormat struct {
	Caches map[string]singleCache
}

type singleCache struct {
	Values map[string][]byte
}

func (f *modifyableFormat) writeItem(section string, key string, value []byte) {
	f.mu.Lock()
	defer f.mu.Unlock()
	_, exists := f.Caches[section]
	if !exists {
		f.Caches[section] = singleCache{}
	}
	f.Caches[section].Values[key] = value
}

func (f *modifyableFormat) readItem(section string, key string) []byte {
	f.mu.Lock()
	defer f.mu.Unlock()
	_, exists := f.Caches[section]
	if !exists {
		return nil
	}
	return f.Caches[section].Values[key]
}

func (c *Cached) loadFile() error {
	f, err := os.Open(c.CacheFile)
	if err != nil {
		return err
	}
	defer func() {
		if err := f.Close(); err != nil {
			c.Log.Log("err", err)
		}
	}()
	var into fileFormat
	if err := json.NewDecoder(f).Decode(&into); err != nil {
		return err
	}
	c.loadedFile.mu.Lock()
	defer c.loadedFile.mu.Unlock()
	c.loadedFile.fileFormat = into
	return nil
}

func (c *Cached) writeBack() error {
	f, err := ioutil.TempFile("", "dcache")
	if err != nil {
		return err
	}
	c.loadedFile.mu.Lock()
	defer c.loadedFile.mu.Unlock()
	if err := json.NewEncoder(f).Encode(c.loadedFile.fileFormat); err != nil {
		return err
	}
	if err := f.Close(); err != nil {
		return err
	}
	return os.Rename(f.Name(), c.CacheFile)
}

func (c *Cached) populateKey(key string) {
	c.loaded.Do(func() {
		if err := c.loadFile(); err != nil {
			c.Log.Log("err", err)
		}
	})
	for _, reader := range c.PopulateFrom.Readers {
		if _, isCache := reader.(*Cached); isCache {
			continue
		}
		readBytes, err := reader.Get(key)
		if err != nil || readBytes == nil {
			continue
		}
		c.loadedFile.writeItem(c.CacheSection, key, readBytes)
		if err := c.writeBack(); err != nil {
			c.Log.Log("err", err)
		}
		return
	}
}

// Get instantly returns the cached value and repopulates from the given distconf
func (c *Cached) Get(key string) ([]byte, error) {
	go c.populateKey(key)
	return c.loadedFile.readItem(c.CacheSection, key), nil
}

// Close just writes back the stored file
func (c *Cached) Close() {
	if err := c.writeBack(); err != nil {
		c.Log.Log("err", err)
	}
}

var _ distconf.Reader = &Cached{}
