package distcache

import (
	"bytes"
	"encoding/json"
	"io/ioutil"
	"os"
	"sync"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/log"
)

type atomicMap struct {
	mu      sync.RWMutex
	results map[string][]byte
}

func (r *atomicMap) get(key string) ([]byte, bool) {
	r.mu.RLock()
	defer r.mu.RUnlock()
	a, b := r.results[key]
	if !b {
		return nil, false
	}
	return a, true
}

func (r *atomicMap) put(key string, value []byte) {
	r.mu.Lock()
	defer r.mu.Unlock()
	if r.results == nil {
		r.results = make(map[string][]byte)
	}
	if value == nil {
		r.results[key] = nil
		return
	}
	toPut := make([]byte, len(value))
	copy(toPut, value)
	r.results[key] = toPut
}

func (r *atomicMap) loadFromFile(filename string) error {
	r.mu.Lock()
	defer r.mu.Unlock()
	fileBytes, err := ioutil.ReadFile(filename)
	if err != nil {
		if os.IsNotExist(err) {
			return nil
		}
		return err
	}
	return json.NewDecoder(bytes.NewReader(fileBytes)).Decode(&r.results)
}

func (r *atomicMap) flushToFile(filename string) error {
	r.mu.RLock()
	defer r.mu.RUnlock()
	tfile1, err := ioutil.TempFile("", "distcache")
	if err != nil {
		return err
	}
	if err := json.NewEncoder(tfile1).Encode(r.results); err != nil {
		return err
	}
	if err := tfile1.Close(); err != nil {
		return err
	}
	if err := os.Remove(filename); err != nil && !os.IsNotExist(err) {
		return err
	}
	return os.Rename(tfile1.Name(), filename)
}

// DistCache helps cache locally distconf values so that local integration tests and startup quickly
type DistCache struct {
	LocalFilename string
	Fallback      distconf.Reader
	Logger        log.Logger
	atomicMap     atomicMap
	doRefresh     chan string
	doFlush       chan struct{}
	onClose       chan struct{}
	once          sync.Once
}

// Get fetches from cache then fallback. If there is a cache value, it also verifies the value is correct
func (d *DistCache) Get(key string) ([]byte, error) {
	d.init()
	ret, exists := d.atomicMap.get(key)
	if exists {
		d.triggerRefresh(key)
		return ret, nil
	}
	result, err := d.Fallback.Get(key)
	if err != nil {
		return nil, err
	}
	d.atomicMap.put(key, result)
	d.triggerFlush()
	return result, nil
}

// Close ends a previously called Start
func (d *DistCache) Close() {
	d.init()
	close(d.onClose)
}

func (d *DistCache) triggerRefresh(key string) {
	select {
	case d.doRefresh <- key:
	default:
	}
}

func (d *DistCache) triggerFlush() {
	select {
	case d.doFlush <- struct{}{}:
	default:
	}
}

func (d *DistCache) log(keyvals ...interface{}) {
	if d.Logger != nil {
		d.Logger.Log(keyvals...)
	}
}

func (d *DistCache) init() {
	d.once.Do(func() {
		d.doRefresh = make(chan string, 1024)
		d.doFlush = make(chan struct{}, 1)
		d.onClose = make(chan struct{})
		if err := d.atomicMap.loadFromFile(d.LocalFilename); err != nil {
			d.log("err", err, "unable to load from local file")
		}
	})
}

// Start begins the refreshing loop
func (d *DistCache) Start() error {
	d.init()
	for {
		select {
		case <-d.onClose:
			return nil
		case <-d.doFlush:
			if err := d.atomicMap.flushToFile(d.LocalFilename); err != nil {
				d.log("err", err, "filename", d.LocalFilename, "unable to flush local file")
			}
		case keyToRefresh := <-d.doRefresh:
			currentValue, exists := d.atomicMap.get(keyToRefresh)
			if !exists {
				continue
			}
			realValue, err := d.Fallback.Get(keyToRefresh)
			if err != nil {
				d.log("err", err, "key", keyToRefresh, "error checking cache is still fresh")
				continue
			}
			if !bytes.Equal(currentValue, realValue) {
				d.log("key", keyToRefresh, "cached", string(currentValue), "real", string(realValue), "cached value does not match source of truth!")
				d.atomicMap.put(keyToRefresh, realValue)
				d.triggerFlush()
			}
		}
	}
}

var _ distconf.Reader = &DistCache{}
