package dynamocursor

import (
	"encoding/base64"
	"encoding/json"
	"fmt"

	"code.justin.tv/hygienic/errors"
)

// Cursor encodes individual cursors for items in a dynamodb query/scan.  You can extract artificial LastEvaluatedKeys
// from it
type Cursor struct {
	TablePartitionValue string `json:"tp"`
	TableSortValue      string `json:"ts"`
	IndexPartitionValue string `json:"ip"`
	IndexSortValue      string `json:"is"`
}

// IsZero returns true if the Cursor is the empty object
func (c Cursor) IsZero() bool {
	return c.TablePartitionValue == "" &&
		c.TableSortValue == "" &&
		c.IndexPartitionValue == "" &&
		c.IndexSortValue == ""
}

func (c Cursor) String() string {
	return fmt.Sprintf("key1(%s/%s) key2 (%s/%s)", c.TablePartitionValue, c.TableSortValue, c.IndexPartitionValue, c.IndexSortValue)
}

// URLEncode is a helper to encode the cursor as a URL safe object
func (c Cursor) URLEncode() string {
	if c.IsZero() {
		return ""
	}
	bs, err := json.Marshal(c)
	if err != nil {
		panic(fmt.Sprintf("Logic error.  Errors should be impossible here: %s", err))
	}
	return base64.URLEncoding.EncodeToString(bs)
}

// URLDecode is a helper to decode the cursor generated from URLEncode
func (c *Cursor) URLDecode(s string) error {
	if s == "" {
		*c = Cursor{}
		return nil
	}

	bs, err := base64.URLEncoding.DecodeString(s)
	if err != nil {
		return errors.Wrapf(err, "failed to base64 decode cursor: %s", s)
	}

	if err := json.Unmarshal(bs, &c); err != nil {
		return errors.Wrapf(err, "failed to unmarshal cursor: %s", string(bs))
	}
	return nil
}
