package dynamocursor_test

import (
	"context"
	"encoding/json"
	"testing"
	"time"

	"io/ioutil"

	"strings"

	"code.justin.tv/hygienic/dynamocursor"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/cep21/circuit"
)

type testConfig struct {
	TableName string `json:"table_name"`
	GsiName   string `json:"gsi_name"`
	Region    string `json:"region"`
	Profile   string `json:"profile"`
	LsiName   string `json:"lsi_name"`
}

func loadConfig(t *testing.T) testConfig {
	var c testConfig
	out, err := ioutil.ReadFile("terraform/testing/test_info.json")
	if err != nil {
		t.Fatal("unable to open testing JSON info", err)
	}
	if err := json.Unmarshal(out, &c); err != nil {
		t.Fatal("unable to unmarshal testing JSON info", err)
	}
	return c
}

func makeSession(t *testing.T) (*session.Session, testConfig) {
	c := loadConfig(t)
	opts := session.Options{
		SharedConfigState: session.SharedConfigEnable,
		Profile:           c.Profile,
		Config: aws.Config{
			Region: &c.Region,
		},
	}
	awsSessionForRole, err := session.NewSessionWithOptions(opts)
	if err != nil {
		t.Skip("unable to get AWS session.  Skipping test")
	}
	return awsSessionForRole, c
}

func TestFactory(t *testing.T) {
	s, cfg := makeSession(t)
	if s == nil {
		return
	}
	m := circuit.Manager{}

	client := dynamodb.New(s)
	f := &dynamocursor.Factory{
		TableNames: func() []string {
			return []string{cfg.TableName}
		},
		Circuit: m.MustCreateCircuit("factory", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout: time.Second * 30,
			},
		}),
		Client: client,
	}
	if err := f.Refresh(context.Background()); err != nil {
		if strings.Contains(err.Error(), "AccessDeniedException") {
			t.Skip("Skipping test since you don't have aws permissions")
		}
		t.Fatal("unable to load table info", err)
	}
	insertData(t, client, cfg)
	a, b := json.Marshal(f)
	t.Log(string(a), b)
	tryQuery(t, f, client, cfg)
	tryScan(t, f, client, cfg)
}

func insertData(t *testing.T, client *dynamodb.DynamoDB, cfg testConfig) {
	data := []map[string]*dynamodb.AttributeValue{
		{
			"user_id": {
				N: aws.String("1"),
			},
			"alias": {
				S: aws.String("jack"),
			},
			"alias_key": {
				B: []byte("hello_key"),
			},
		},
		{
			"user_id": {
				N: aws.String("1"),
			},
			"alias": {
				S: aws.String("john"),
			},
			"alias_key": {
				B: []byte("hello_key2"),
			},
		},
		{
			"user_id": {
				N: aws.String("1"),
			},
			"alias": {
				S: aws.String("bill"),
			},
			"alias_key": {
				B: []byte("hello_key2"),
			},
		},
		{
			"user_id": {
				N: aws.String("1"),
			},
			"alias": {
				S: aws.String("jim"),
			},
			"alias_key": {
				B: []byte("hello_key3"),
			},
		},
		{
			"user_id": {
				N: aws.String("2"),
			},
			"alias": {
				S: aws.String("jim"),
			},
			"alias_key": {
				B: []byte("hello_key2"),
			},
		},
		{
			"user_id": {
				N: aws.String("3"),
			},
			"alias": {
				S: aws.String("jim"),
			},
			"alias_key": {
				B: []byte("hello_key2"),
			},
		},
		{
			"user_id": {
				N: aws.String("4"),
			},
			"alias": {
				S: aws.String("jim"),
			},
			"alias_key": {
				B: []byte("hello_key2"),
			},
		},
	}
	for _, d := range data {
		_, err := client.PutItem(&dynamodb.PutItemInput{
			TableName: &cfg.TableName,
			Item:      d,
		})
		if err != nil {
			t.Fatal("unable to put item", err)
		}
	}
}

func tryScan(t *testing.T, f *dynamocursor.Factory, client *dynamodb.DynamoDB, cfg testConfig) {
	var cursor dynamocursor.Cursor
	es, err := f.ExclusiveStartKey(cursor, cfg.TableName, cfg.GsiName)
	if err != nil {
		t.Fatal("unable to generate exclusive start key", err)
	}
	if es != nil {
		t.Fatal("Expect no start key at the beginning")
	}
	res, err := client.Scan(&dynamodb.ScanInput{
		TableName:         &cfg.TableName,
		IndexName:         &cfg.GsiName,
		Limit:             aws.Int64(3),
		ExclusiveStartKey: es,
	})
	if err != nil {
		t.Fatal("could not query input", err)
	}
	if len(res.Items) != 3 {
		t.Fatal("need 3 items to work")
	}
	shouldRefetchLastTwoScan(t, res, f, client, cfg)
}

func tryQuery(t *testing.T, f *dynamocursor.Factory, client *dynamodb.DynamoDB, cfg testConfig) {
	var cursor dynamocursor.Cursor
	es, err := f.ExclusiveStartKey(cursor, cfg.TableName, cfg.GsiName)
	if err != nil {
		t.Fatal("unable to generate exclusive start key", err)
	}
	if es != nil {
		t.Fatal("Expect no start key at the beginning")
	}
	res, err := client.Query(&dynamodb.QueryInput{
		TableName: &cfg.TableName,
		IndexName: &cfg.GsiName,
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":alias": {
				S: aws.String("jim"),
			},
		},
		KeyConditionExpression: aws.String("alias=:alias"),
		Limit:                  aws.Int64(3),
		ExclusiveStartKey:      es,
	})
	if err != nil {
		t.Fatal("could not query input", err)
	}
	if len(res.Items) != 3 {
		t.Fatal("need 3 items to work")
	}
	shouldRefetchLastTwo(t, res, f, client, cfg)
}

func shouldRefetchLastTwo(t *testing.T, res *dynamodb.QueryOutput, f *dynamocursor.Factory, client *dynamodb.DynamoDB, cfg testConfig) {
	newCursor := mustCursor(t, f, res.Items[0], cfg.TableName, cfg.GsiName)
	esk, err := f.ExclusiveStartKey(newCursor, cfg.TableName, cfg.GsiName)
	if err != nil {
		t.Fatal("unable to get esk", err)
	}
	res2, err := client.Query(&dynamodb.QueryInput{
		TableName: &cfg.TableName,
		IndexName: &cfg.GsiName,
		Limit:     aws.Int64(3),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":alias": {
				S: aws.String("jim"),
			},
		},
		KeyConditionExpression: aws.String("alias=:alias"),
		ExclusiveStartKey:      esk,
	})
	if err != nil {
		t.Fatal("unable to query last two", err, newCursor)
	}
	if len(res2.Items) != 3 {
		t.Fatal("expect 3 new items")
	}
	inputLastTwo := [2]dynamocursor.Cursor{
		mustCursor(t, f, res.Items[1], cfg.TableName, cfg.GsiName),
		mustCursor(t, f, res.Items[2], cfg.TableName, cfg.GsiName),
	}
	outputFirstTwo := [2]dynamocursor.Cursor{
		mustCursor(t, f, res2.Items[0], cfg.TableName, cfg.GsiName),
		mustCursor(t, f, res2.Items[1], cfg.TableName, cfg.GsiName),
	}
	for i := 0; i < len(inputLastTwo); i++ {
		if inputLastTwo[i] != outputFirstTwo[i] {
			t.Fatal("expected input and output to match at index ", i, inputLastTwo[i], outputFirstTwo[i])
		}
	}
}

func shouldRefetchLastTwoScan(t *testing.T, res *dynamodb.ScanOutput, f *dynamocursor.Factory, client *dynamodb.DynamoDB, cfg testConfig) {
	newCursor := mustCursor(t, f, res.Items[0], cfg.TableName, cfg.GsiName)
	esk, err := f.ExclusiveStartKey(newCursor, cfg.TableName, cfg.GsiName)
	if err != nil {
		t.Fatal("unable to get esk", err)
	}
	res2, err := client.Scan(&dynamodb.ScanInput{
		TableName:         &cfg.TableName,
		IndexName:         &cfg.GsiName,
		Limit:             aws.Int64(3),
		ExclusiveStartKey: esk,
	})
	if err != nil {
		t.Fatal("unable to scan last two", err, newCursor)
	}
	if len(res2.Items) != 3 {
		t.Fatal("expect 3 new items")
	}
	inputLastTwo := [2]dynamocursor.Cursor{
		mustCursor(t, f, res.Items[1], cfg.TableName, cfg.GsiName),
		mustCursor(t, f, res.Items[2], cfg.TableName, cfg.GsiName),
	}
	outputFirstTwo := [2]dynamocursor.Cursor{
		mustCursor(t, f, res2.Items[0], cfg.TableName, cfg.GsiName),
		mustCursor(t, f, res2.Items[1], cfg.TableName, cfg.GsiName),
	}
	for i := 0; i < len(inputLastTwo); i++ {
		if inputLastTwo[i] != outputFirstTwo[i] {
			t.Fatal("expected input and output to match at index ", i, inputLastTwo[i], outputFirstTwo[i])
		}
	}
}

func mustCursor(t *testing.T, f *dynamocursor.Factory, item map[string]*dynamodb.AttributeValue, tableName string, indexName string) dynamocursor.Cursor {
	ret, err := f.Cursor(item, tableName, indexName)
	if err != nil {
		t.Fatal("unable to generate cursor", err)
	}
	return ret
}
