package elastimemcache_test

import (
	"context"
	"encoding/json"
	"math"
	"os"
	"strconv"
	"testing"
	"time"

	"code.justin.tv/hygienic/elastimemcache"
	"code.justin.tv/hygienic/gomemcache/memcache"
)

var _ memcache.ServerSelector = &elastimemcache.Elasticache{}

func testReads(t *testing.T, c *memcache.Client) {
	for i := 0; i < 100; i++ {
		asStr := strconv.Itoa(i)
		err := c.Set(context.Background(), &memcache.Item{
			Key:   asStr,
			Value: []byte(strconv.Itoa(i + 1)),
		})
		if err != nil {
			t.Fatal("expected no error from set", err)
		}
	}
}

func testWrites(t *testing.T, c *memcache.Client) {
	for i := 0; i < 100; i++ {
		asStr := strconv.Itoa(i)
		item, err := c.Get(context.Background(), asStr)
		if err != nil {
			t.Fatal("expected no error from set", err)
		}
		asInt, err := strconv.ParseInt(string(item.Value), 10, 64)
		if err != nil {
			t.Fatal("expected an int back from memcache", err)
		}
		if asInt != int64(i+1) {
			t.Fatal("expected i+1 back from i", asInt, "!=", i+1)
		}
	}
}

func testRandomness(t *testing.T, e *elastimemcache.Elasticache) {
	picksByAddr := map[string]int64{}
	for i := 0; i < 1000; i++ {
		asStr := strconv.Itoa(i)
		addr, err := e.PickServer(asStr)
		if err != nil {
			t.Fatal("expected to pick a server", err)
		}
		picksByAddr[addr.String()] = picksByAddr[addr.String()] + 1
	}
	smallest := int64(math.MaxInt64)
	largest := int64(0)
	for _, val := range picksByAddr {
		if smallest > val {
			smallest = val
		}
		if largest < val {
			largest = val
		}
	}
	t.Log("smallest", smallest, "largest", largest)
	if smallest*4 < largest {
		t.Fatal("spread between smallest and largest hits per server too large")
	}
}

func TestWithServer(t *testing.T) {
	cfgEndpoint := os.Getenv("ELASTICACHE_SERVER")
	if cfgEndpoint == "" {
		t.Skip("Skipping test without ELASTICACHE_SERVER variable set")
	}

	t.Log("using ELASTICACHE_SERVER", cfgEndpoint)
	e := &elastimemcache.Elasticache{
		CfgServer: cfgEndpoint,
	}
	ctx := context.Background()
	if err := e.Init(ctx); err != nil {
		t.Fatal("unable to init from endpoint", err)
	}
	version, servers := e.Configuration()
	if len(servers) == 0 {
		t.Fatal("I expected some servers!")
	}
	t.Log("using configuration", version, servers)
	testRandomness(t, e)

	c := memcache.NewFromSelector(e)
	c.Timeout = time.Second
	c.MaxIdleConns = 100

	testReads(t, c)
	testWrites(t, c)
	b, err := json.Marshal(c.ClientStats())
	if err != nil {
		t.Fatal("expected to marshal client stats")
	}
	t.Log("stats are", string(b), err)
	b, err = json.Marshal(e)
	if err != nil {
		t.Fatal("expected to marshal easticache keeper")
	}
	t.Log("elasticache is", string(b), err)
}
