package goform

import (
	"encoding/base64"
	"encoding/json"
	"fmt"
	"github.com/awslabs/goformation/cloudformation"
	"github.com/awslabs/goformation/intrinsics"
	"github.com/sanathkr/yaml"
	"io"
	"reflect"
	"strings"
	"sync/atomic"
)

const lazyNameIntrinsic = "lazyname"

// Parameter are https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/parameters-section-structure.html
type Parameter struct {
	AllowedPattern        string   `json:",omitempty"`
	AllowedValues         []string `json:",omitempty"`
	ConstraintDescription string   `json:",omitempty"`
	Default               *string  `json:",omitempty"`
	Description           string   `json:",omitempty"`
	MaxLength             int      `json:",omitempty"`
	MaxValue              int      `json:",omitempty"`
	MinLength             int      `json:",omitempty"`
	MinValue              int      `json:",omitempty"`
	NoEcho                bool     `json:",omitempty"`
	Type                  string
}

type InjectableResource interface {
	AddToTemplate(t *WrappedTemplate) error
}

// Named allows references to the names of resources that aren't attached yet
type named struct {
	template *WrappedTemplate
	resource interface{}
}

var _ json.Marshaler = &named{}

func (r *named) MarshalJSON() ([]byte, error) {
	s, err := r.template.Name(r.resource)
	if err != nil {
		return nil, err
	}
	return []byte(s), nil
}

// OutputExport is Export of https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html
type OutputExport struct {
	Name string `json:",omitempty"`
}

// Output are https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html
type Output struct {
	Description string `json:",omitempty"`
	Value       string
	Export      *OutputExport `json:"Export,omitempty"`
}

// WrappedTemplate wraps a goformation cloudformation template
type WrappedTemplate struct {
	*cloudformation.Template
	idx int32
	lazyNameIdx int32
	lazyNameMap map[int32]interface{}
}

// NewTemplate creates a new wrapped goformation template
func NewTemplate() *WrappedTemplate {
	return &WrappedTemplate{
		Template: cloudformation.NewTemplate(),
	}
}

func usableName(s string) string {
	parts := strings.Split(s, ".")
	if len(parts) > 0 {
		s = parts[len(parts) - 1]
	}
	return strings.Map(func(r rune) rune {
		if (r >= 'A' && r <= 'Z') || (r >= 'a' && r <='z') {
			return r
		}
		return -1
	}, s)
}

// Select returns a single object from a list of objects, where the list itself is a single item
func SelectArr(index string, list string) string {
	return encode(`{ "Fn::Select": [ "` + index + `", "` + list + `" ] }`)
}

// MustAdd injects the resource into the template or panics if there is an error
func (w *WrappedTemplate) MustAdd(r InjectableResource) {
	if err := r.AddToTemplate(w); err != nil {
		panic(err)
	}
}

// AttachResource adds a resource with a generated name.
func (w *WrappedTemplate) AttachResource(resource interface{}) interface{} {
	name := fmt.Sprintf("%03d%s", atomic.AddInt32(&w.idx, 1), usableName(reflect.TypeOf(resource).String()))
	return w.AttachResourceWithName(resource, name)
}

// AttachParameter adds a parameter to the template
func (w *WrappedTemplate) AttachParameter(name string, param *Parameter) *Parameter {
	w.Template.Parameters[name] = param
	return param
}

// AttachParameter adds a parameter to the template
func (w *WrappedTemplate) AttachOutput(name string, output *Output) *Output {
	w.Template.Outputs[name] = output
	return output
}

//AttachResourceWithName is like AttachResource but with a specific name
func (w *WrappedTemplate) AttachResourceWithName(resource interface{}, name string) interface{} {
	w.Resources[name] = resource
	return resource
}

// MustName finds the name of a resource or panics
func (w *WrappedTemplate) MustName(resource interface{}) string {
	ret, err := w.Name(resource)
	if err != nil {
		panic(err)
	}
	return ret
}

//// Or returns true if any one of the specified conditions evaluate to true, or returns false if all of the conditions evaluates to false. Fn::Or acts as an OR operator. The minimum number of conditions that you can include is 2, and the maximum is 10.
//func Or(conditions []string) string {
//	return encode(`{ "Fn::Or": [ "` + strings.Trim(strings.Join(conditions, `", "`), `, "`) + `" ] }`)
//}
//
// encode takes a string representation of an intrinsic function, and base64 encodes it.
// This prevents the escaping issues when nesting multiple layers of intrinsic functions.
func encode(value string) string {
	return base64.StdEncoding.EncodeToString([]byte(value))
}

// JSON converts an AWS CloudFormation template object to JSON
func (w *WrappedTemplate) JSON() ([]byte, error) {

	j, err := json.MarshalIndent(w.Template, "", "  ")
	if err != nil {
		return nil, err
	}

	return intrinsics.ProcessJSON(j, &intrinsics.ProcessorOptions{
		IntrinsicHandlerOverrides: map[string]intrinsics.IntrinsicHandler {
			lazyNameIntrinsic: w.intrinsicName,
		},
	})
}

// YAML converts an AWS CloudFormation template object to YAML
func (w *WrappedTemplate) YAML() ([]byte, error) {
	j, err := w.JSON()
	if err != nil {
		return nil, err
	}
	return yaml.JSONToYAML(j)
}

// MustYAMLStdout prints the template as YAML to stdout and panics if any failures happen
func (w *WrappedTemplate) MustYAMLOut(out io.Writer) {
	b, err := w.YAML()
	if err != nil {
		panic(err)
	}
	if _, err := io.WriteString(out, string(b)); err != nil {
		panic(err)
	}
}

// Named allows references to the names of resources that aren't attached yet.  We need a direct name of something, but
// we want it to reference
func (w *WrappedTemplate) Named(resource interface{}) string {
	thisIdx := atomic.AddInt32(&w.lazyNameIdx, 1)
	if w.lazyNameMap == nil {
		w.lazyNameMap = map[int32]interface{}{}
	}
	w.lazyNameMap[thisIdx] = resource
	l := map[string]map[string]int32 {
		"Ref": {
			lazyNameIntrinsic: thisIdx,
		},
	}
	ret, err := json.Marshal(l)
	if err != nil {
		// Unexpected
		panic(err)
	}
	return encode(string(ret))
}

// Name finds the name of a resource or returns an error
func (w *WrappedTemplate) Name(resource interface{}) (string, error) {
	for name, r := range w.Resources {
		if r == resource {
			return name, nil
		}
	}
	for name, p := range w.Parameters {
		if p == resource {
			return name, nil
		}
	}
	return "", fmt.Errorf("unable to find resource %+v", resource)
}

// MustRef is the same as "!Ref <resource>"
func (w *WrappedTemplate) MustRef(resource interface{}) string {
	return cloudformation.Ref(w.MustName(resource))
}

func (w *WrappedTemplate) intrinsicName(name string, input interface{}, template interface{}) interface{} {
	// Check the input is a string
	src, ok := input.(float64)
	if !ok {
		panic("expected a string as input")
	}

	return w.MustName(w.lazyNameMap[int32(src)])
}