package gometrics

import (
	"errors"
	"runtime"
	"time"
)

// StatSender sends stats
type StatSender interface {
	Inc(string, int64, float32) error
	Gauge(string, int64, float32) error
	TimingDuration(string, time.Duration, float32) error
}

// Logger logs values
type Logger interface {
	Log(vals ...interface{})
}

// Service sets up periodic gometrics stat sending
type Service struct {
	Statter  StatSender
	Interval time.Duration
	Logger   Logger
	onDone   chan struct{}
	onClose  chan struct{}
}

// Setup the service
func (s *Service) Setup() error {
	if s.Statter == nil {
		return errors.New("expected Statter to be non-nil")
	}

	if s.Interval <= 0 {
		s.Interval = 5 * time.Second
	}

	s.onDone = make(chan struct{})
	s.onClose = make(chan struct{})

	return nil
}

// Start sending gometric stats
func (s *Service) Start() error {
	defer close(s.onDone)

	var mstats runtime.MemStats
	runtime.ReadMemStats(&mstats)
	ticker := time.NewTicker(s.Interval)
	defer ticker.Stop()

	for {
		select {
		case <-ticker.C:
			mstats = s.send(mstats)
		case <-s.onClose:
			return nil
		}
	}
}

// Close stops sending gometric stats
func (s *Service) Close() error {
	close(s.onClose)
	<-s.onDone
	return nil
}

func (s *Service) send(previous runtime.MemStats) runtime.MemStats {
	var mstats runtime.MemStats
	runtime.ReadMemStats(&mstats)

	// goroutines
	s.gauge("go.num_goroutines", int64(runtime.NumGoroutine()))

	// objects
	s.inc("go.mem.mallocs", int64(mstats.Mallocs-previous.Mallocs))
	s.inc("go.mem.frees", int64(mstats.Frees-previous.Frees))
	s.gauge("go.mem.sys", int64(mstats.Sys))
	s.gauge("go.mem.alloc", int64(mstats.Alloc))

	// gc
	s.inc("go.gc.collections", int64(mstats.NumGC-previous.NumGC))
	s.gauge("go.gc.percent_cpu", fractionToPercent(mstats.GCCPUFraction))
	s.timingDuration("go.gc.last_duration", time.Duration(mstats.PauseNs[(mstats.NumGC+255)%256]))
	s.timingDuration("go.gc.total_duration", time.Duration(mstats.PauseTotalNs-previous.PauseTotalNs))

	// heap
	s.gauge("go.mem.heap_objects", int64(mstats.HeapObjects))
	s.gauge("go.mem.heap.sys", int64(mstats.HeapSys))
	s.gauge("go.mem.heap.idle", int64(mstats.HeapIdle))
	s.gauge("go.mem.heap.inuse", int64(mstats.HeapInuse))
	s.gauge("go.mem.heap.released", int64(mstats.HeapReleased))

	// stack
	s.gauge("go.mem.stack.sys", int64(mstats.StackSys))
	s.gauge("go.mem.stack.inuse", int64(mstats.StackInuse))

	// other
	s.gauge("go.mem.mspan.sys", int64(mstats.MSpanSys))
	s.gauge("go.mem.mcache.sys", int64(mstats.MCacheSys))
	s.gauge("go.mem.buckhash.sys", int64(mstats.BuckHashSys))
	s.gauge("go.mem.other.sys", int64(mstats.OtherSys))

	return mstats
}

func (s *Service) gauge(metric string, val int64) {
	s.onError(s.Statter.Gauge(metric, val, 1.0), "sending gauge stat")
}

func (s *Service) inc(metric string, val int64) {
	s.onError(s.Statter.Inc(metric, val, 1.0), "sending counter stat")
}

func (s *Service) timingDuration(metric string, val time.Duration) {
	s.onError(s.Statter.TimingDuration(metric, val, 1.0), "sending timing stat")
}

func (s *Service) onError(err error, msg string) {
	if err != nil {
		s.log("err", err, msg)
	}
}

func (s *Service) log(vals ...interface{}) {
	if s.Logger != nil {
		s.Logger.Log(vals...)
	}
}

func fractionToPercent(fraction float64) int64 {
	return int64((fraction * 100) + 0.5)
}
