package httpheaders

import "net/http"

// Client is the minimal http interface we require
type Client interface {
	Do(req *http.Request) (*http.Response, error)
}

// HeaderClient adds a header name to a http client
type HeaderClient struct {
	Client      Client
	HeaderName  string
	HeaderValue string
}

var _ Client = &http.Client{}
var _ Client = &HeaderClient{}

// Do adds the header to the request, then removes it when done
func (h *HeaderClient) Do(req *http.Request) (*http.Response, error) {
	client := h.Client
	if client == nil {
		client = http.DefaultClient
	}
	return setupHeaders(req, h.HeaderName, h.HeaderValue, client.Do)
}

// WithHeader constructs a HeaderClient with the default http client if required
func WithHeader(client Client, headerName string, headerValue string) *HeaderClient {
	return &HeaderClient{
		Client:      client,
		HeaderName:  headerName,
		HeaderValue: headerValue,
	}
}

func setupHeaders(req *http.Request, headerName string, headerValue string, doFunc func(req *http.Request) (*http.Response, error)) (*http.Response, error) {
	if headerName == "" {
		return doFunc(req)
	}
	prevHeader := req.Header.Get(headerName)
	defer func() {
		if prevHeader == "" {
			req.Header.Del(headerName)
		} else {
			req.Header.Set(headerName, prevHeader)
		}
	}()
	req.Header.Set(headerName, headerValue)
	return doFunc(req)
}
