package messagebatch

import (
	"context"
	"time"
)

// Logger is the minimal interface for logging methods
type Logger interface {
	Log(keyvals ...interface{})
}

// Config controls how to configure your Batch message sender
type Config struct {
	// Default time.Second
	FlushWaitTime time.Duration
	// Default 1024
	FlushSize int64
}

// Batcher allows sending batch messages
type Batcher struct {
	// Optional logger
	Log Logger
	// Optional config
	Config     *Config
	SendEvents func([]interface{}) error
	Events     chan interface{}
	onClose    chan struct{}
}

// Setup should be called before Start or Close
func (a *Batcher) Setup() error {
	a.onClose = make(chan struct{})
	return nil
}

func (a *Batcher) log(keyvals ...interface{}) {
	if a.Log != nil {
		a.Log.Log(keyvals...)
	}
}

// Event adds an event
func (a *Batcher) Event(event interface{}) {
	select {
	case a.Events <- event:
	default:
		a.log("event buffer full")
	}
}

func (a *Batcher) flushSize() int64 {
	if a.Config.FlushSize == 0 {
		return 1024
	}
	return a.Config.FlushSize
}

func (a *Batcher) flushWaitTime() time.Duration {
	if a.Config.FlushWaitTime == 0 {
		return time.Second
	}
	return a.Config.FlushWaitTime
}

// EmptyEvents empties out any leftover events and returns.  Returns early if ctx dies.
func (a *Batcher) EmptyEvents(ctx context.Context) error {
	events := make([]interface{}, 0, a.flushSize())
	for {
		events = events[0:0]
		select {
		case event := <-a.Events:
			events = append(events, event)
		case <-ctx.Done():
			return ctx.Err()
		default:
			return nil
		}
		waitTime := time.After(a.flushWaitTime())
	fillBuffer:
		for len(events) < int(a.flushSize()) {
			select {
			case <-a.onClose:
				break fillBuffer
			case event := <-a.Events:
				events = append(events, event)
			case <-waitTime:
				break fillBuffer
			}
		}
		if err := a.SendEvents(events); err != nil {
			a.log("err", err, "unable to send events")
		}
	}
}

// Start sending added events.  Should run in a goroutine
func (a *Batcher) Start() error {
	events := make([]interface{}, 0, a.flushSize())
	for {
		events = events[0:0]
		select {
		case <-a.onClose:
			return nil
		case event := <-a.Events:
			events = append(events, event)
		}
		waitTime := time.After(a.flushWaitTime())
	fillBuffer:
		for len(events) < int(a.flushSize()) {
			select {
			case <-a.onClose:
				break fillBuffer
			case event := <-a.Events:
				events = append(events, event)
			case <-waitTime:
				break fillBuffer
			}
		}
		if err := a.SendEvents(events); err != nil {
			a.log("err", err, "unable to send events")
		}
	}
}

// Close ends Start
func (a *Batcher) Close() error {
	close(a.onClose)
	return nil
}
