package metricscactusstatsd

import (
	"reflect"
	"testing"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	"code.justin.tv/hygienic/metrics"
	"github.com/stretchr/testify/assert"
)

var tPid = identifier.ProcessIdentifier{
	Service:  "MyService",
	Stage:    "prod",
	Substage: "primary",
	Region:   "us-west-2",
	Machine:  "SomeUniqueMachineIdentifier",
}

func TestStatsd_NewDimensionalSubStatter(t *testing.T) {
	base := Statsd{}
	sub := base.NewDimensionalSubStatter(map[string]string{"second": "value"})
	inner := sub.(*Statsd)
	assert.Equal(t, map[string]string{"second": "value"}, inner.dimensions)
}

func TestStatsd_NewSubStatter(t *testing.T) {
	base := Statsd{}
	sub := base.NewSubStatter("substatter")
	inner := sub.(*Statsd)
	assert.Equal(t, inner.metricName("key"), "substatter.key")
}

func TestStatsd_metricDimensions(t *testing.T) {
	type fields struct {
		Registry   metrics.BaseRegistry
		prefix     string
		dimensions map[string]string
	}
	type args struct {
		dimensions map[string]string
		metricType string
	}
	tests := []struct {
		name   string
		fields fields
		args   args
		want   map[string]string
	}{
		{
			name: "Nil dimensions",
			args: args{dimensions: nil, metricType: "counter"},
			want: map[string]string{"MetricType": "counter"},
		},
		{
			name:   "Overridden dimension",
			fields: fields{dimensions: map[string]string{"BasicInfo": "base"}},
			args:   args{dimensions: map[string]string{"BasicInfo": "new"}, metricType: "counter"},
			want:   map[string]string{"BasicInfo": "new", "MetricType": "counter"},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			s := &Statsd{
				Registry:   tt.fields.Registry,
				prefix:     tt.fields.prefix,
				dimensions: tt.fields.dimensions,
			}
			if got := s.metricDimensions(tt.args.dimensions, tt.args.metricType); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Statsd.metricDimensions() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestTelemetryStatsdShim_NewDimensionalSubStatter(t *testing.T) {
	base := NewTelemetryStatsdShim(nil, "", map[string]string{}, tPid)
	sub := base.NewDimensionalSubStatter(map[string]string{"second": "value"})
	inner := sub.(*telemetryStatsdShim)
	assert.Equal(t, map[string]string{"second": "value"}, inner.dimensions)
}

func TestTelemetryStatsdShim_NewSubStatter(t *testing.T) {
	base := NewTelemetryStatsdShim(nil, "", map[string]string{}, tPid)
	sub := base.NewSubStatter("substatter")
	inner := sub.(*telemetryStatsdShim)
	assert.Equal(t, "substatter.key", inner.metricName("key"))
}

func TestTelemetryStatsdShim_metricDimensions(t *testing.T) {
	type fields struct {
		Registry   metrics.BaseRegistry
		prefix     string
		dimensions map[string]string
	}
	type args struct {
		dimensions map[string]string
		metricType string
	}
	tests := []struct {
		name   string
		fields fields
		args   args
		want   map[string]string
	}{
		{
			name: "Nil dimensions",
			args: args{dimensions: nil, metricType: "counter"},
			want: map[string]string{"MetricType": "counter"},
		},
		{
			name:   "Overridden dimension",
			fields: fields{dimensions: map[string]string{"BasicInfo": "base"}},
			args:   args{dimensions: map[string]string{"BasicInfo": "new"}, metricType: "counter"},
			want:   map[string]string{"BasicInfo": "new", "MetricType": "counter"},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			s := NewTelemetryStatsdShim(tt.fields.Registry, tt.fields.prefix, tt.fields.dimensions, tPid)
			if got := s.(*telemetryStatsdShim).metricDimensions(tt.args.dimensions, tt.args.metricType); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("TelemetryStatsdShim.metricDimensions() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestNewTelemetryStatterShim(t *testing.T) {
	base := NewTelemetryStatterShim(nil, "", map[string]string{}, tPid)
	sub := base.NewDimensionalSubStatter(map[string]string{"second": "value"})
	inner := sub.(*telemetryStatsdShim)
	assert.Equal(t, map[string]string{"second": "value"}, inner.dimensions)
}

func TestTelemetryStatterShim_SetPrefix(t *testing.T) {
	base := NewTelemetryStatterShim(nil, "zeroth", map[string]string{}, tPid)
	base.SetPrefix("first")
	sub := base.NewSubStatter("substatter")
	inner := sub.(*telemetryStatsdShim)
	assert.Equal(t, "first.substatter.key", inner.metricName("key"))
}

func TestTelemetryStatterShim_Close(t *testing.T) {
	base := NewTelemetryStatterShim(nil, "zeroth", map[string]string{}, tPid)
	assert.NoError(t, base.Close())
}