package metrics

import "context"

// TimeWindowAggregation is an aggregation of values inside a time window
type TimeWindowAggregation struct {
	Va ValueAggregation
	Tw TimeWindow
}

// Union Merges two aggregations inside a time period
func (a TimeWindowAggregation) Union(other TimeWindowAggregation) TimeWindowAggregation {
	return TimeWindowAggregation{
		Va: a.Va.Union(other.Va),
		Tw: a.Tw.Union(other.Tw),
	}
}

// Record a value. The terminology is borrowed from https://godoc.org/golang.org/x/net/internal/timeseries#Observable
// And https://github.com/prometheus/client_golang
type Observer interface {
	Observe(value float64)
}

type Aggregator interface {
	MetricCollector
	Observer
}

type nopAggregator struct {
}

func (n *nopAggregator) CollectMetrics() []TimeWindowAggregation {
	return nil
}

func (n *nopAggregator) Observe(value float64) {
}

var _ Aggregator = &nopAggregator{}

type MetadataConstructor func(TimeSeriesIdentifier, TimeSeriesMetadata) TimeSeriesMetadata
type AggregationConstructor func(ts *TimeSeries) Aggregator

type TimeSeriesSource interface {
	TimeSeries(tsi TimeSeriesIdentifier, metadata MetadataConstructor) *TimeSeries
}

type BaseRegistry interface {
	// Get a unique time series for some meta data
	TimeSeriesSource
	Observer(ts *TimeSeries) Observer
	// Set a metric collector for a time series
	GetOrSet(ts *TimeSeries, mc MetricCollectorConstructor) MetricCollector
}

type OnDemandFlushable interface {
	CurrentMetrics(r TimeSeriesSource) []TimeSeriesAggregation
}

// ValueAggregator is anything that can observe values and return aggregations
type ValueAggregator interface {
	Observer
	Aggregate() ValueAggregation
}

type AggregationSink interface {
	Aggregate(context.Context, []TimeSeriesAggregation) error
}

type AggregationSource interface {
	FlushMetrics() []TimeSeriesAggregation
}

type MetricCollector interface {
	CollectMetrics() []TimeWindowAggregation
}

type MetricCollectorConstructor func(ts *TimeSeries) MetricCollector

type TimeSeriesAggregation struct {
	TS          *TimeSeries
	Aggregation TimeWindowAggregation
}
