package metricsext

import (
	"context"
	"sync"
	"time"

	"code.justin.tv/hygienic/metrics"
)

type Logger interface {
	Log(kvs ...interface{})
}

type Flushable interface {
	Flush(ctx context.Context) error
}

type AggregationFlusher struct {
	Source metrics.AggregationSource
	Sink   metrics.AggregationSink
}

func (f *AggregationFlusher) Flush(ctx context.Context) error {
	return f.Sink.Aggregate(ctx, f.Source.FlushMetrics())
}

var _ Flushable = &AggregationFlusher{}

type PeriodicFlusher struct {
	Flushable Flushable

	// Optional
	FlushTimeout time.Duration
	Interval     time.Duration
	Logger       Logger
	TimeAfter    func(time.Duration) <-chan time.Time

	once             sync.Once
	startEverStarted bool
	startDone        chan struct{}
	onClose          chan struct{}
}

func (f *PeriodicFlusher) interval() time.Duration {
	if f.Interval == 0 {
		return time.Minute
	}
	return f.Interval
}

func (f *PeriodicFlusher) setup(inStart bool) {
	f.once.Do(func() {
		f.onClose = make(chan struct{})
		f.startDone = make(chan struct{})
		f.startEverStarted = inStart
	})
}

func (f *PeriodicFlusher) flushWithContext() error {
	ctx := context.Background()
	if f.FlushTimeout != 0 {
		var onDone context.CancelFunc
		ctx, onDone = context.WithTimeout(ctx, f.FlushTimeout)
		defer onDone()
	}
	return f.Flushable.Flush(ctx)
}

func (f *PeriodicFlusher) Start() error {
	f.setup(true)
	defer close(f.startDone)
	for {
		select {
		case <-f.onClose:
			if err := f.flushWithContext(); err != nil {
				f.log("err", err)
			}
			return nil
		case <-f.timeAfter(f.interval()):
			if err := f.flushWithContext(); err != nil {
				f.log("err", err)
			}
		}
	}
}

func (f *PeriodicFlusher) Close() error {
	f.setup(false)
	close(f.onClose)
	if f.startEverStarted {
		<-f.startDone
	}
	return nil
}

func (f *PeriodicFlusher) log(kvs ...interface{}) {
	if f.Logger != nil {
		f.Logger.Log(kvs...)
	}
}

func (f *PeriodicFlusher) timeAfter(interval time.Duration) <-chan time.Time {
	if f.TimeAfter == nil {
		return time.After(interval)
	}
	return f.TimeAfter(interval)
}
