package objectcache

import (
	"math/rand"
	"reflect"
	"strconv"
	"testing"

	"github.com/stretchr/testify/require"
)

type compressionRun struct {
	name         string
	ValueDecoder ValueDecoder
	object       map[string]string
}

func largeObject() map[string]string {
	ret := make(map[string]string, 2000)
	for i := 0; i < 2000; i++ {
		ret[strconv.Itoa(i)] = strconv.Itoa(rand.Int())
	}
	return ret
}

func BenchmarkCompression(b *testing.B) {
	b.ReportAllocs()
	runs := []compressionRun{
		{
			name:         "gzip",
			ValueDecoder: GZipDecoder(),
			object:       largeObject(),
		},
		{
			name:         "json",
			ValueDecoder: defaultDecoder,
			object:       largeObject(),
		},
	}
	for _, run := range runs {
		b.Run(run.name, func(b *testing.B) {
			b.ResetTimer()
			b.ReportAllocs()
			for i := 0; i < b.N; i++ {
				_, err := run.ValueDecoder.Marshal(run.object)
				require.NoError(b, err)
			}
		})
	}
}

func TestCompression(t *testing.T) {
	runs := []compressionRun{
		{
			name:         "gzip",
			ValueDecoder: GZipDecoder(),
			object:       largeObject(),
		},
		{
			name:         "json",
			ValueDecoder: defaultDecoder,
			object:       largeObject(),
		},
	}
	for _, run := range runs {
		t.Run(run.name, func(t *testing.T) {
			compressed, err := run.ValueDecoder.Marshal(run.object)
			require.NoError(t, err)
			var into map[string]string
			err = run.ValueDecoder.Unmarshal(compressed, &into)
			require.NoError(t, err)
			require.True(t, reflect.DeepEqual(into, run.object))
			require.True(t, len(into) > 0)
		})
	}
}

func BenchmarkDecompression(b *testing.B) {
	b.ReportAllocs()
	runs := []compressionRun{
		{
			name:         "gzip",
			ValueDecoder: GZipDecoder(),
			object:       largeObject(),
		},
		{
			name:         "json",
			ValueDecoder: defaultDecoder,
			object:       largeObject(),
		},
	}
	for _, run := range runs {
		b.Run(run.name, func(b *testing.B) {
			res, err := run.ValueDecoder.Marshal(run.object)
			require.NoError(b, err)
			b.ReportAllocs()
			for i := 0; i < b.N; i++ {
				into := make(map[string]string)
				err := run.ValueDecoder.Unmarshal(res, &into)
				require.NoError(b, err)
			}
		})
	}
}
