package rollbar

import (
	"bytes"
	"context"
	"encoding/json"
	"expvar"
	"fmt"
	"hash/adler32"
	"hash/crc32"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"os/user"
	"reflect"
	"runtime"
	"strings"
	"sync/atomic"
	"time"
)

// DefaultConfiguration is used when no configuration is specified.  It's usually OK to ignore this, but it's public
// if you really want to change it.
var DefaultConfiguration = DataOptionals{
	Platform: runtime.GOOS,
	Language: "go",
	Notifier: Notifier{
		Name:    LibraryName,
		Version: Version,
	},
}

var currentUser *user.User

func init() {
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown<err>"
	}
	currentUser, err = user.Current()
	if err != nil {
		currentUser = nil
	}
	DefaultConfiguration.Server.Host = hostname
	DefaultConfiguration.CodeVersion = CodeVersion
}

// CodeVersion can use the linker to set this variable (if desired)
var CodeVersion string

const (
	// LibraryName is the name of this library: sent with the request
	LibraryName = "rollbar2"
	// Version of this library: sent with the request
	Version = "0.0.1"
	// DefaultEndpoint is where the library sends data by default
	DefaultEndpoint = "https://api.rollbar.com/api/1/item/"
	// DefaultDeploymentEndpoint is where to send new deployment information
	DefaultDeploymentEndpoint = "https://api.rollbar.com/api/1/deploy/"
	// DefaultDrainSize is the default HTTP connection draining size
	DefaultDrainSize = 1024
	// DefaultEnvironment is sent if no env is specified in the client
	DefaultEnvironment = "development"
)

// Client connects to and sends rollbar values
type Client struct {
	// AccessToken is required and is used to authenticate the request
	AccessToken string
	// Environment is required and defaults to DefaultEnvironment
	Environment string

	// Enpoint is optional and defaults to rollbar
	Endpoint string
	// HTTPClient is optional and used to send the request
	HTTPClient http.Client
	// MessageDefaults are optional and allow you to specify default values with the request.  If you change
	// this you probably want to merge it with DefaultConfiguration
	MessageDefaults *DataOptionals
	// DrainSize is optional and used to drain the HTTP connection after it is sent
	DrainSize int64
	// Now is optional and used to get the current time
	Now func() time.Time

	stats Stats
}

// Response is the result of a message send
type Response struct {
	Err    int            `json:"err"`
	Result ResponseResult `json:"result"`
}

// ResponseResult is part of rollbar's response object.  See rollbar documentation for what it means.
type ResponseResult struct {
	UUID string `json:"uuid"`
}

func (c *Client) url() string {
	if c.Endpoint != "" {
		return c.Endpoint
	}
	return DefaultEndpoint
}

// Stats return internal stat tracking
func (c *Client) Stats() Stats {
	return c.stats.Clone()
}

// Var returns an expvar about the client
func (c *Client) Var() expvar.Var {
	return expvar.Func(func() interface{} {
		return map[string]interface{}{
			"Environment":     c.Environment,
			"Endpoint":        c.Endpoint,
			"Stats":           c.Stats(),
			"MessageDefaults": c.MessageDefaults,
		}
	})
}

func (c *Client) now() time.Time {
	if c.Now == nil {
		return time.Now()
	}
	return c.Now()
}

func (c *Client) maxDrainSize() int64 {
	if c.DrainSize != 0 {
		return c.DrainSize
	}
	return DefaultDrainSize
}

func (c *Client) environment() string {
	if c.Environment != "" {
		return c.Environment
	}
	return DefaultEnvironment
}

// MessageBody returns a request body that can be used for message requests.  Send it with the Send method.
func MessageBody(level Level, msg string) *Data {
	return &Data{
		Body: &Body{
			Message: &Message{
				Body: msg,
			},
		},
		DataOptionals: DataOptionals{
			Level: level,
		},
	}
}

// TraceBody returns a message body that can be used for Trace requests.  Send it with the Send method.
func TraceBody(level Level, err error) *Data {
	d := &Data{
		Body: &Body{
			Trace: &Trace{
				Frames: buildStack(1, err),
				Exception: Exception{
					Class:   errorClass(err),
					Message: err.Error(),
				},
			},
		},
		DataOptionals: DataOptionals{
			Level: level,
		},
	}
	d.Fingerprint = calculateFingerprint(d.Body.Trace.Frames, err, true)
	return d
}

// Message sends a message to rollbar.
func (c *Client) Message(ctx context.Context, level Level, msg string) (*Response, error) {
	d := MessageBody(level, msg)
	return c.Send(ctx, d)
}

// Trace sends an error to Rollbar.  If the error object implements StackTrace() []uintptr then that stack trace will be
// used instead.
func (c *Client) Trace(ctx context.Context, level Level, err error) (*Response, error) {
	d := TraceBody(level, err)
	return c.Send(ctx, d)
}

// Deployment sends an error to Rollbar.  If the error object implements StackTrace() []uintptr then that stack trace will be
// used instead.
func (c *Client) Deployment(ctx context.Context, revision string, options *DeploymentOptions) error {
	if options == nil {
		options = &DeploymentOptions{}
	}
	atomic.AddInt64(&c.stats.DeploymentsSent, 1)
	body := deploymentPostBody{
		LocalUsername:   options.LocalUsername,
		RollbarUsername: options.RollbarUsername,
		Comment:         options.Comment,
		Revision:        revision,
		AccessToken:     c.AccessToken,
	}
	if options.Environment != nil {
		body.Environment = *options.Environment
	}
	if body.LocalUsername == nil && currentUser != nil {
		body.LocalUsername = &currentUser.Username
	}
	if body.Environment == "" {
		body.Environment = c.Environment
	}
	jsonBody := &bytes.Buffer{}
	if err := json.NewEncoder(jsonBody).Encode(body); err != nil {
		return err
	}
	req, err := http.NewRequest("POST", DefaultDeploymentEndpoint, jsonBody)
	if err != nil {
		return err
	}
	req.Header.Set("Content-Type", "application/json")
	req = req.WithContext(ctx)
	resp, err := c.HTTPClient.Do(req)
	if err != nil {
		atomic.AddInt64(&c.stats.SentHTTPErrors, 1)
		return err
	}
	decodeInto := deploymentResponse{}
	return c.drainBody(resp, &decodeInto)
}

func (c *Client) setupRequest(ctx context.Context, d *Data) (*http.Response, error) {
	d.MergeFrom(c.MessageDefaults)
	d.Timestamp = c.now().Unix()
	toSend := itemToSend{
		AccessToken: c.AccessToken,
		Data: sendData{
			Environment: c.environment(),
			Data:        d,
		},
	}
	jsonBody := &bytes.Buffer{}
	if err := json.NewEncoder(jsonBody).Encode(toSend); err != nil {
		return nil, err
	}
	req, err := http.NewRequest("POST", c.url(), jsonBody)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Content-Type", "application/json")
	req = req.WithContext(ctx)
	resp, err := c.HTTPClient.Do(req)
	if err != nil {
		return nil, err
	}
	return resp, nil
}

func (c *Client) drainBody(resp *http.Response, decodeInto interface{}) (retErr error) {
	defer func() {
		_, e2 := io.CopyN(ioutil.Discard, resp.Body, c.maxDrainSize())
		if e2 != nil && e2 != io.EOF && retErr == nil {
			retErr = e2
		}
		e3 := resp.Body.Close()
		if e3 != nil && retErr == nil {
			retErr = e3
		}
	}()
	if resp.StatusCode != http.StatusOK {
		badBody := bytes.Buffer{}
		_, err := io.CopyN(&badBody, resp.Body, c.maxDrainSize())
		return &responseErr{
			body:     badBody,
			httpcode: resp.StatusCode,
			wrapped:  err,
		}
	}
	if err := json.NewDecoder(resp.Body).Decode(decodeInto); err != nil {
		return err
	}
	return nil
}

// Send a constructed Body.  You may want to use Trace or Message as they are usually easier to interact with.
func (c *Client) Send(ctx context.Context, d *Data) (sendResponse *Response, retErr error) {
	var resp *http.Response
	var err error
	if d.Body.Message != nil {
		atomic.AddInt64(&c.stats.MessagesSent, 1)
	}
	if d.Body.Trace != nil {
		atomic.AddInt64(&c.stats.TracesSent, 1)
	}
	if resp, err = c.setupRequest(ctx, d); err != nil {
		atomic.AddInt64(&c.stats.SentHTTPErrors, 1)
		return nil, err
	}
	sendResponse = &Response{}
	if retErr = c.drainBody(resp, sendResponse); retErr != nil {
		atomic.AddInt64(&c.stats.SentHTTPErrors, 1)
		return nil, retErr
	}
	if sendResponse.Err != 0 {
		atomic.AddInt64(&c.stats.SentRollbarErrors, 1)
		return sendResponse, &responseErr{
			responseCode: sendResponse.Err,
		}
	}
	return sendResponse, nil
}

func cause(err error) error {
	type causer interface {
		Cause() error
		error
	}
	if cerr, ok := err.(causer); ok {
		return cerr
	}
	return err
}

func errorClass(err error) string {
	err = cause(err)
	class := reflect.TypeOf(err).String()
	if class == "" {
		return "panic"
	} else if class == "*errors.errorString" {
		checksum := adler32.Checksum([]byte(err.Error()))
		return fmt.Sprintf("{%x}", checksum)
	} else {
		return strings.TrimPrefix(class, "*")
	}
}

// StackFrameFingerprint is a way to calculate a unique string from a stack trace: used by rollbar to group logs
func StackFrameFingerprint(frames []Frame) string {
	return calculateFingerprint(frames, nil, false)
}

func calculateFingerprint(frames []Frame, err error, useErrString bool) string {
	err = cause(err)
	hash := crc32.NewIEEE()

	// If the Error() string contains random junk, we can signal inside the error what to hash by.  Note we still
	// use the stack trace, so sometimes returning "" is enough
	type errFingerprint interface {
		Fingerprint() string
	}
	if fp, ok := err.(errFingerprint); ok {
		fmt.Fprint(hash, fp.Fingerprint())
	} else if useErrString {
		fmt.Fprint(hash, err.Error())
	}
	for _, frame := range frames {
		fmt.Fprintf(hash, "%s%s%d", frame.Filename, frame.Method, frame.Line)
	}
	return fmt.Sprintf("%x", hash.Sum32())

}
