// +build integration

package rollbar

import (
	"errors"
	"io/ioutil"
	"runtime"
	"strings"
	"testing"

	"context"

	. "github.com/smartystreets/goconvey/convey"
)

func MustLoadToken() string {
	b, err := ioutil.ReadFile(".rollbar_token")
	if err != nil {
		panic("Unable to run integration tests without rollbar token in file .rollbar_token")
	}
	return strings.TrimSpace(string(b))
}

type errWithTrace struct {
	error
	trace []uintptr
}

func (e *errWithTrace) Stack() []uintptr {
	return e.trace
}

func newErrWithTrace(msg string) error {
	const depth = 32
	var pcs [depth]uintptr
	n := runtime.Callers(2, pcs[:])
	st := pcs[0:n]
	return &errWithTrace{
		error: errors.New(msg),
		trace: st,
	}
}

func TestIntegrationClient(t *testing.T) {
	Convey("With a client", t, func() {
		c := Client{
			AccessToken: MustLoadToken(),
		}
		ctx := context.Background()
		Convey("Must be able to send with a code version", func() {
			newDefs := DefaultConfiguration
			newDefs.CodeVersion = "v1.0"
			c.MessageDefaults = &newDefs
			resp, err := c.Message(ctx, Debug, "TestCodeVersion")
			So(err, ShouldBeNil)
			So(resp.Result.UUID, ShouldNotEqual, "")
		})
		Convey("Must be able to send a message", func() {
			resp, err := c.Message(ctx, Debug, "TestIntegrationClient_msg")
			So(err, ShouldBeNil)
			So(resp.Result.UUID, ShouldNotEqual, "")
		})
		Convey("Must be able to send info message", func() {
			resp, err := c.Message(ctx, Info, "Should_be_info_msg")
			So(err, ShouldBeNil)
			So(resp.Result.UUID, ShouldNotEqual, "")
		})
		Convey("Must be able to send war message", func() {
			resp, err := c.Message(ctx, Warning, "Should_be_war_msg")
			So(err, ShouldBeNil)
			So(resp.Result.UUID, ShouldNotEqual, "")
		})
		Convey("Must be able to send trace", func() {
			resp, err := c.Trace(ctx, Debug, newErrWithTrace("TestIntegrationClientTrace_trace"))
			So(err, ShouldBeNil)
			So(resp.Result.UUID, ShouldNotEqual, "")
		})
		Convey("Must be able to send deployment information", func() {
			c.Environment = "staging"
			err := c.Deployment(ctx, "237d1e7b82e1ee6462a5752dda1947e85ae819a2", nil)
			So(err, ShouldBeNil)
		})
		Convey("Defaults should work", func() {
			newDefaults := DefaultConfiguration
			newDefaults.CodeVersion = "abcde"
			newDefaults.Custom = map[string]string{
				"tag1": "val1",
			}
			newDefaults.Server.Host = "override_host"
			c.MessageDefaults = &newDefaults
			resp, err := c.Trace(ctx, Debug, newErrWithTrace("TestIntegrationClientTrace_newdefaults"))
			So(err, ShouldBeNil)
			So(resp.Result.UUID, ShouldNotEqual, "")
		})
	})
}
