package rollbar

import (
	"bytes"
	"errors"
	"io"
	"net/http"
	"net/http/httptest"
	"testing"

	"context"

	. "github.com/smartystreets/goconvey/convey"
)

func ShouldContainSubstrings(cvy C, s string, parts ...string) {
	for _, p := range parts {
		cvy.So(s, ShouldContainSubstring, p)
	}
}

func TestRollbarClient(t *testing.T) {
	Convey("With a rollbar client", t, func() {
		var h http.HandlerFunc
		s := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
			h.ServeHTTP(w, req)
		}))
		c := Client{
			Endpoint:    s.URL,
			AccessToken: "abc",
		}
		ctx := context.Background()

		defaultBodySend := func(cvy C, w http.ResponseWriter, req *http.Request) string {
			cvy.So(req.Method, ShouldEqual, "POST")
			cvy.So(req.Header.Get("Content-Type"), ShouldEqual, "application/json")
			_, err := w.Write([]byte("{}"))
			cvy.So(err, ShouldBeNil)
			var body bytes.Buffer
			_, err = io.Copy(&body, req.Body)
			cvy.So(err, ShouldBeNil)
			return body.String()
		}

		Convey("Should send messages", func(cvy C) {
			h = http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
				defaultBodySend(cvy, w, req)
			})
			resp, err := c.Message(ctx, Debug, "hello")
			So(err, ShouldBeNil)
			So(resp.Err, ShouldEqual, 0)
		})

		Convey("Should check error codes", func(cvy C) {
			h = http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
				w.WriteHeader(http.StatusNotFound)
				_, err := w.Write([]byte("{}"))
				cvy.So(err, ShouldBeNil)
			})
			resp, err := c.Message(ctx, Debug, "hello")
			So(err.Error(), ShouldContainSubstring, "404")
			So(resp, ShouldBeNil)
		})

		Convey("Should check error in body", func(cvy C) {
			h = http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
				_, err := w.Write([]byte(`{"Err": 1101}`))
				cvy.So(err, ShouldBeNil)
			})
			_, err := c.Message(ctx, Debug, "hello")
			So(err.Error(), ShouldContainSubstring, "1101")
		})

		Convey("Should send body params", func(cvy C) {
			toSend := MessageBody(Debug, "hello1121")

			h = http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
				body := defaultBodySend(cvy, w, req)
				ShouldContainSubstrings(cvy, body, toSend.Custom["akey"], "hello1121", toSend.Server.Host)
			})

			toSend.Custom = map[string]string{
				"akey": "avalue",
			}
			toSend.Server.Host = "helloworldserver"
			_, err := c.Send(ctx, toSend)
			So(err, ShouldBeNil)
		})

		Convey("Should allow defaults", func(cvy C) {
			newDefaults := &DataOptionals{
				CodeVersion: "abcdefg",
				Custom: map[string]string{
					"appname": "myservice",
				},
			}
			newDefaults.MergeFrom(&DefaultConfiguration)
			c.MessageDefaults = newDefaults

			h = http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
				body := defaultBodySend(cvy, w, req)
				ShouldContainSubstrings(cvy, body, newDefaults.Custom["akey"], newDefaults.Custom["appname"])
			})
			_, err := c.Message(ctx, Debug, "hello")
			So(err, ShouldBeNil)

			Convey("and should merge defaults", func(cvy C) {
				msgBody := MessageBody(Warning, "helloaaba")
				msgBody.Custom = map[string]string{
					"hello": "world1234",
				}

				h = http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
					body := defaultBodySend(cvy, w, req)
					ShouldContainSubstrings(cvy, body, newDefaults.Custom["akey"], msgBody.Custom["hello"], "helloaaba")
				})
				_, err := c.Send(ctx, msgBody)
				So(err, ShouldBeNil)
			})
		})

		Convey("Should send errors", func(cvy C) {
			h = http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
				body := defaultBodySend(cvy, w, req)
				ShouldContainSubstrings(cvy, body, "a new error", "client_test.go")
			})

			errToSend := errors.New("a new error")
			_, err := c.Trace(ctx, Debug, errToSend)
			So(err, ShouldBeNil)
		})

		Reset(func() {
			s.Close()
		})
	})
}
