package seh1

import (
	"math"

	"code.justin.tv/hygienic/metrics"
	"code.justin.tv/hygienic/metrics/metricsext"
)

var defaultConstants HashConstants

func init() {
	// Note: The original implementation implied this value has to be used.  That's not true, it's a MWS abstraction bleed.
	//       Even with MWS, we could still change this value and work backwards.  We should find the right value for twitch.

	// Note: without this, our structure is not correct for extreme float values
	defaultConstants = ComputeHashConstants(math.Log(1.1))
}

func (h *HashConstants) smallestValidBucket() int32 {
	smallestValidBucket := compute(math.SmallestNonzeroFloat64, h.BucketFactor)
	for {
		reverseBucket := compute(inverse(smallestValidBucket, h.BucketFactor), h.BucketFactor)
		if reverseBucket == smallestValidBucket {
			return reverseBucket
		}
		if reverseBucket < smallestValidBucket {
			// This is strange .. panic(?)
			panic("Inf loop?")
		}
		smallestValidBucket = reverseBucket
	}
}

func (h *HashConstants) biggestValidBucket() int32 {
	biggestValidBucket := compute(math.MaxFloat64, h.BucketFactor)
	for {
		reverseBucket := compute(inverse(biggestValidBucket, h.BucketFactor), h.BucketFactor)
		if reverseBucket == biggestValidBucket {
			return reverseBucket
		}
		if reverseBucket > biggestValidBucket {
			// This is strange .. panic(?)
			panic("Inf loop?")
		}
		biggestValidBucket = reverseBucket
	}
}

// ComputeHashConstants creates a SHA1 HashConstants from a bucket value
func ComputeHashConstants(factor float64) HashConstants {
	ret := HashConstants{
		BucketFactor: factor,
	}
	ret.SmallestValidBucket = ret.smallestValidBucket()
	ret.BiggestValidBucket = ret.biggestValidBucket()
	return ret
}

// HashConstants are used by SEH1 to detect valid bucket ranges
type HashConstants struct {
	BucketFactor float64
	// This number minus 1 is 0.
	// For values smaller than this, SEH1 will produce incorrect results since we cannot inverse buckets correctly
	SmallestValidBucket int32
	// This number +1 is +inf.  Since we always 'floor', no buckets should be bigger than this
	// For values bigger than this, SEH1 will produce incorrect results since we cannot inverse buckets correctly
	BiggestValidBucket int32
}

// Go from bucket index to an approx value
func (h HashConstants) computeBucket(v float64) int32 {
	ret := compute(v, h.BucketFactor)
	if ret < h.SmallestValidBucket {
		return h.SmallestValidBucket - 1
	}
	if ret >= h.BiggestValidBucket {
		return h.BiggestValidBucket
	}
	return ret
}

// Go from bucket index to an approx value
func (h HashConstants) inverseBucket(bucketIndex int32) float64 {
	if bucketIndex < h.SmallestValidBucket {
		return 0
	}
	if bucketIndex > h.BiggestValidBucket {
		return math.MaxFloat64
	}
	return inverse(bucketIndex, h.BucketFactor)
}

// SEH1 is a sample aggregator that produces a sparse exponential histogram.  Default bucket factor is log(1.1)
type SEH1 struct {
	HashConstants HashConstants

	// map of bucket index to counts at that bucket index.
	positiveValues map[int32]int32
	negativeValues map[int32]int32

	// buckets for special values that otherwise cause issues
	zeroBucket  int32
	posInfinity int32
	negInfinity int32
	nan         int32
}

var _ metrics.Bucketer = &SEH1{}

func (h *SEH1) hashConstants() HashConstants {
	if h.HashConstants.BucketFactor == 0 {
		return defaultConstants
	}
	return h.HashConstants
}

// Buckets returns all the bucketed values for this histogram
func (h *SEH1) Buckets() []metrics.Bucket {
	ret := make([]metrics.Bucket, 0, len(h.positiveValues)+len(h.negativeValues))
	// Set the individual values
	for bucketVal, sampleCount := range h.positiveValues {
		ret = append(ret, metrics.Bucket{
			Start: h.hashConstants().inverseBucket(bucketVal),
			End:   h.hashConstants().inverseBucket(bucketVal + 1),
			Count: sampleCount,
		})
	}

	for bucketVal, sampleCount := range h.negativeValues {
		ret = append(ret, metrics.Bucket{
			Start: -h.hashConstants().inverseBucket(bucketVal + 1),
			End:   -h.hashConstants().inverseBucket(bucketVal),
			Count: sampleCount,
		})
	}
	if h.zeroBucket > 0 {
		ret = append(ret, metrics.Bucket{
			Start: 0,
			End:   0,
			Count: h.zeroBucket,
		})
	}
	if h.posInfinity > 0 {
		ret = append(ret, metrics.Bucket{
			Start: math.Inf(1),
			End:   math.Inf(1),
			Count: h.posInfinity,
		})
	}
	if h.negInfinity > 0 {
		ret = append(ret, metrics.Bucket{
			Start: math.Inf(-1),
			End:   math.Inf(-1),
			Count: h.negInfinity,
		})
	}
	if h.nan > 0 {
		ret = append(ret, metrics.Bucket{
			Start: math.NaN(),
			End:   math.NaN(),
			Count: h.nan,
		})
	}
	return ret
}

// Observe adds a sample to the histogram.
func (h *SEH1) Observe(value float64) {
	if value == 0 {
		h.zeroBucket++
		return
	}
	if math.IsInf(value, 1) {
		h.posInfinity++
		return
	}
	if math.IsInf(value, -1) {
		h.negInfinity++
		return
	}
	if math.IsNaN(value) {
		h.nan++
		return
	}
	if value > 0 {
		h.observeIn(value, &h.positiveValues)
	} else {
		h.observeIn(-value, &h.negativeValues)
	}
}

// Observe adds a sample to the histogram.
func (h *SEH1) observeIn(value float64, hist *map[int32]int32) {
	if *hist == nil {
		*hist = make(map[int32]int32)
	}
	if value < 0 {
		panic("Logic error")
	}
	// Can't log negative values lol
	bucket := h.hashConstants().computeBucket(value)
	// Note: histogram grows without bound.  You may think we need to limit size of this, but the value space grows
	//       exponentially so the total range of valid buckets is actually tiny.
	(*hist)[bucket]++
}

func inverse(idx int32, factor float64) float64 {
	return math.Pow(math.E, float64(idx)*factor)
}

func compute(v float64, factor float64) int32 {
	return int32(math.Floor(math.Log(v) / factor))
}

// SEHRollingAggregation returns the same SEH rolling aggregation for each time series
func SEHRollingAggregation(_ *metrics.TimeSeries) metrics.Aggregator {
	return &metricsext.RollingAggregation{
		AggregatorFactory: func() metrics.ValueAggregator {
			return &metricsext.LocklessValueAggregator{
				Bucketer: &SEH1{},
			}
		},
	}
}
