package seh1

import (
	"fmt"
	"math"
	"math/rand"
	"reflect"
	"sort"
	"testing"

	"code.justin.tv/hygienic/metrics"
)

func TestSEH1_Buckets(t *testing.T) {
	type fields struct {
		BucketFactor   float64
		positiveValues map[int32]int32
		negativeValues map[int32]int32
		zeroBucket     int32
	}
	tests := []struct {
		name   string
		fields fields
		want   []metrics.Bucket
	}{
		{
			name: "empty",
			want: []metrics.Bucket{},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			h := &SEH1{
				positiveValues: tt.fields.positiveValues,
				negativeValues: tt.fields.negativeValues,
				zeroBucket:     tt.fields.zeroBucket,
			}
			if got := h.Buckets(); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("SEH1.Buckets() = %v, want %v", got, tt.want)
			}
		})
	}
}

func sortBuckets(in []metrics.Bucket) []metrics.Bucket {
	sort.SliceStable(in, func(i, j int) bool {
		if math.IsNaN(in[i].Start) {
			return true
		}
		return in[i].Start < in[j].Start || (in[i].Start == in[j].Start && in[i].End < in[j].End)
	})
	return in
}

func TestSEH1_ObserveMany(t *testing.T) {
	tests := []struct {
		name string
		args []float64
		res  []metrics.Bucket
	}{
		{
			name: "1 2 3",
			args: []float64{1, 2, 3},
			res: []metrics.Bucket{
				{
					Count: 1,
					Start: 1,
					End:   1.1,
				}, {
					Count: 1,
					Start: 1.948717100000001,
					End:   2.143588810000001,
				}, {
					Count: 1,
					Start: 2.853116706110002,
					End:   3.1384283767210026,
				},
			},
		}, {
			name: "1 2 3 0",
			args: []float64{0, 1, 2, 3},
			res: []metrics.Bucket{
				{
					Count: 1,
					Start: 0,
					End:   0,
				}, {
					Count: 1,
					Start: 1,
					End:   1.1,
				}, {
					Count: 1,
					Start: 1.948717100000001,
					End:   2.143588810000001,
				}, {
					Count: 1,
					Start: 2.853116706110002,
					End:   3.1384283767210026,
				},
			},
		}, {
			name: "1 2 -1 -2 0",
			args: []float64{1, 2, -1, -2, 0},
			res: []metrics.Bucket{
				{
					Count: 1,
					Start: -2.143588810000001,
					End:   -1.948717100000001,
				}, {
					Count: 1,
					Start: -1.1,
					End:   -1,
				}, {
					Count: 1,
					Start: 0,
					End:   0,
				}, {
					Count: 1,
					Start: 1,
					End:   1.1,
				}, {
					Count: 1,
					Start: 1.948717100000001,
					End:   2.143588810000001,
				},
			},
		}, {
			name: "NaN",
			args: []float64{math.NaN()},
			res: []metrics.Bucket{
				{
					Count: 1,
					Start: math.NaN(),
					End:   math.NaN(),
				},
			},
		}, {
			name: "biggest",
			args: []float64{math.MaxFloat64},
			res: []metrics.Bucket{
				{
					Count: 1,
					Start: 1.7837187326220709e+308,
					End:   math.MaxFloat64,
				},
			},
		}, {
			name: "tiny",
			args: []float64{-math.SmallestNonzeroFloat64},
			res: []metrics.Bucket{
				{
					Count: 1,
					Start: -1.935434747225168e-308,
					End:   0,
				},
			},
		}, {
			name: "smallest",
			args: []float64{-math.MaxFloat64},
			res: []metrics.Bucket{
				{
					Count: 1,
					Start: -math.MaxFloat64,
					End:   -1.7837187326220709e+308,
				},
			},
		}, {
			name: "special_values",
			args: []float64{0, 1, -1, math.MaxFloat64, -math.MaxFloat64, math.SmallestNonzeroFloat64, -math.SmallestNonzeroFloat64, math.Inf(1), math.Inf(-1), math.NaN()},
			res: []metrics.Bucket{
				{
					Count: 1,
					Start: math.Inf(-1),
					End:   math.Inf(-1),
				}, {
					Count: 1,
					Start: -math.MaxFloat64,
					End:   -1.7837187326220709e+308,
				}, {
					Count: 1,
					Start: -1.935434747225168e-308,
					End:   0,
				}, {
					Count: 1,
					Start: 0,
					End:   0,
				}, {
					Count: 1,
					Start: 0,
					End:   1.935434747225168e-308,
				}, {
					Count: 1,
					Start: 1,
					End:   1.1,
				}, {
					Count: 1,
					Start: -1.1,
					End:   -1,
				}, {
					Count: 1,
					Start: 1.7837187326220709e+308,
					End:   math.MaxFloat64,
				}, {
					Count: 1,
					Start: math.NaN(),
					End:   math.NaN(),
				}, {
					Count: 1,
					Start: math.Inf(1),
					End:   math.Inf(1),
				},
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			var h SEH1
			for _, arg := range tt.args {
				h.Observe(arg)
			}
			got := sortBuckets(h.Buckets())
			tt.res = sortBuckets(tt.res)
			bucketNanEqual(got, tt.res)
			if !reflect.DeepEqual(trimNan(got), trimNan(tt.res)) {
				t.Errorf("SEH1.Buckets() = %v, want %v", got, tt.res)
			}
			bucketsMakeSense(t, tt.args, got)
		})
	}
}

func trimNan(b1 []metrics.Bucket) []metrics.Bucket {
	ret := make([]metrics.Bucket, 0, len(b1))
	for _, b := range b1 {
		if !math.IsNaN(b.Start) {
			ret = append(ret, b)
		}
	}
	return ret
}

func bucketNanEqual(b1 []metrics.Bucket, b2 []metrics.Bucket) {
	aNan := int32(0)
	bNan := int32(0)
	for _, b := range b1 {
		if math.IsNaN(b.Start) {
			aNan += b.Count
		}
	}
	for _, b := range b2 {
		if math.IsNaN(b.Start) {
			bNan += b.Count
		}
	}
	if aNan != bNan {
		panic("NaN not equal")
	}
}

func contains(b metrics.Bucket, v float64) bool {
	return b.Start == v || (v >= b.Start && v <= b.End)
}

func bucketsMakeSense(t *testing.T, vals []float64, buckets []metrics.Bucket) {
vloop:
	for _, v := range vals {
		for i := range buckets {
			if buckets[i].Start == v && buckets[i].End == v && buckets[i].Count > 0 {
				buckets[i].Count--
				continue vloop
			}
			if math.IsNaN(buckets[i].Start) && math.IsNaN(v) && buckets[i].Count > 0 {
				buckets[i].Count--
				continue vloop
			}
		}
		for i := range buckets {
			if contains(buckets[i], v) && buckets[i].Count > 0 {
				buckets[i].Count--
				continue vloop
			}
		}
		t.Errorf("Unable to find value %g in bucket array", v)
	}
	for _, buck := range buckets {
		if buck.Count != 0 {
			t.Errorf("bucket count left %v", buck)
		}
	}
}

func TestSEH1_Observe(t *testing.T) {
	type fields struct {
		BucketFactor   float64
		positiveValues map[int32]int32
		negativeValues map[int32]int32
		zeroBucket     int32
	}
	type args struct {
		value float64
	}
	tests := []struct {
		name   string
		fields fields
		args   args
	}{
		{
			name: "Empty observe won't panic",
			args: args{
				value: 0,
			},
		}, {
			name: "-inf",
			args: args{
				value: math.Inf(-1),
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			h := &SEH1{
				positiveValues: tt.fields.positiveValues,
				negativeValues: tt.fields.negativeValues,
				zeroBucket:     tt.fields.zeroBucket,
			}
			h.Observe(tt.args.value)
		})
	}
}

func TestSEH1_observeIn(t *testing.T) {
	type fields struct {
		BucketFactor   float64
		positiveValues map[int32]int32
		negativeValues map[int32]int32
		zeroBucket     int32
	}
	type args struct {
		value float64
		hist  *map[int32]int32
	}
	h1 := make(map[int32]int32)
	tests := []struct {
		name   string
		fields fields
		args   args
	}{
		{
			name: "empty observe in not panic",
			args: args{
				value: 1,
				hist:  &h1,
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			h := &SEH1{
				positiveValues: tt.fields.positiveValues,
				negativeValues: tt.fields.negativeValues,
				zeroBucket:     tt.fields.zeroBucket,
			}
			h.observeIn(tt.args.value, tt.args.hist)
		})
	}
}

func TestSEH1_inverseBucket(t *testing.T) {
	type fields struct {
		BucketFactor   float64
		positiveValues map[int32]int32
		negativeValues map[int32]int32
		zeroBucket     int32
	}
	type args struct {
		bucketIndex int32
	}
	tests := []struct {
		name   string
		fields fields
		args   args
		want   float64
	}{
		{
			name: "inverse of zero",
			want: 1,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			h := &SEH1{
				positiveValues: tt.fields.positiveValues,
				negativeValues: tt.fields.negativeValues,
				zeroBucket:     tt.fields.zeroBucket,
			}
			if got := h.hashConstants().inverseBucket(tt.args.bucketIndex); got != tt.want {
				t.Errorf("SEH1.inverseBucket() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestSEH1_computeBucket(t *testing.T) {
	type fields struct {
		BucketFactor   float64
		positiveValues map[int32]int32
		negativeValues map[int32]int32
		zeroBucket     int32
	}
	type args struct {
		v float64
	}
	tests := []struct {
		name   string
		fields fields
		args   args
		want   int32
	}{
		{
			name: "Compute of one",
			args: args{
				v: 1,
			},
			want: 0,
		}, {
			name: "smallest",
			args: args{
				v: math.SmallestNonzeroFloat64,
			},
			want: -7435,
		}, {
			name: "biggest",
			args: args{
				v: math.MaxFloat64,
			},
			want: 7447,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			h := &SEH1{
				positiveValues: tt.fields.positiveValues,
				negativeValues: tt.fields.negativeValues,
				zeroBucket:     tt.fields.zeroBucket,
			}
			if got := h.hashConstants().computeBucket(tt.args.v); got != tt.want {
				t.Errorf("SEH1.computeBucket() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestInversableBuckets(t *testing.T) {
	vals := []float64{
		1,
		math.Inf(1),
	}
	for _, r := range vals {
		t.Run(fmt.Sprintf("%v", r), func(t *testing.T) {
			var h SEH1
			bucket := h.hashConstants().computeBucket(r)
			backToVal := h.hashConstants().inverseBucket(bucket)
			backToBucket := h.hashConstants().computeBucket(backToVal)
			if bucket != backToBucket {
				t.Error("Unable to inverse", r, bucket, backToVal, backToBucket, math.Log(r), h.hashConstants().inverseBucket(bucket-1))
			}
		})
	}
}

func TestSEHRollingAggregation(t *testing.T) {
	x := SEHRollingAggregation(nil)
	if x == nil {
		t.Error("Expect non nil")
	}
}

//func TestHashConstants

func TestComputeHashConstants(t *testing.T) {
	type args struct {
		factor float64
	}
	tests := []struct {
		name string
		args args
		want HashConstants
	}{
		{
			name: "Default",
			args: args{
				math.Log(1.1),
			},
			want: HashConstants{
				BucketFactor:        math.Log(1.1),
				SmallestValidBucket: -7434,
				BiggestValidBucket:  7447,
			},
		}, {
			name: "Big factor",
			args: args{
				1,
			},
			want: HashConstants{
				BucketFactor:        1,
				SmallestValidBucket: -709,
				BiggestValidBucket:  709,
			},
		}, {
			name: "Tiny factor",
			args: args{
				math.Log(1.001),
			},
			want: HashConstants{
				BucketFactor:        math.Log(1.001),
				SmallestValidBucket: -708752,
				BiggestValidBucket:  710137,
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := ComputeHashConstants(tt.args.factor); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("ComputeHashConstants() = %v, want %v", got, tt.want)
			}
			verifyConstants(t, tt.want)
		})
	}
}

func verifyConstants(t *testing.T, hc HashConstants) {
	x := compute(inverse(compute(inverse(hc.SmallestValidBucket, hc.BucketFactor), hc.BucketFactor), hc.BucketFactor), hc.BucketFactor)
	if x != hc.SmallestValidBucket {
		t.Errorf("smallest bucket value does not inverse %d", x)
	}

	// See if smaller than smallest works
	x2 := compute(inverse(compute(inverse(hc.SmallestValidBucket-1, hc.BucketFactor), hc.BucketFactor), hc.BucketFactor), hc.BucketFactor)
	if x2 == hc.SmallestValidBucket-1 {
		t.Errorf("We can be even smaller: %d", x2)
	}

	y := compute(inverse(compute(inverse(hc.BiggestValidBucket, hc.BucketFactor), hc.BucketFactor), hc.BucketFactor), hc.BucketFactor)
	if y != hc.BiggestValidBucket {
		t.Errorf("Biggest bucket value does not inverse %d", x)
	}

	// See if bigger than biggest works
	y2 := compute(inverse(compute(inverse(hc.BiggestValidBucket+1, hc.BucketFactor), hc.BucketFactor), hc.BucketFactor), hc.BucketFactor)
	if y2 == hc.BiggestValidBucket+1 {
		t.Errorf("We can be even bigger: %d", y2)
	}
}

func BenchmarkSEH1_Observe(b *testing.B) {
	var h SEH1
	for i := 0; i < b.N; i++ {
		h.Observe(rand.Float64())
	}
}

func TestHashConstants_computeBucket(t *testing.T) {
	type testCase struct {
		name     string
		H        HashConstants
		v        float64
		expected int32
	}
	tests := []testCase{
		{
			name:     "tiny fraction",
			H:        defaultConstants,
			v:        math.SmallestNonzeroFloat64,
			expected: -7435,
		}, {
			name:     "really big number",
			H:        defaultConstants,
			v:        math.MaxFloat64,
			expected: 7447,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got := tt.H.computeBucket(tt.v)
			if got != tt.expected {
				t.Errorf("HashConstants.computeBucket() = %v, want %v", got, tt.expected)
			}
		})
	}
}
