package simplevpn

import (
	"code.justin.tv/hygienic/goform"
	"github.com/awslabs/goformation/cloudformation"
)


// SubnetBlock is the IP block a subnet can have
type SubnetBlock struct {
	PrivateBlock string
	PublicBlock string
}

type Config struct {
	OutputPrefix string
	VpcCidrBlock string
	SubnetBlocks []SubnetBlock
}

// VPC is a simple VPC with basic public and private subnets that are exported for use by other templates
type VPC struct {
	Config Config

	VpcCidrBlock     *goform.Parameter
	VPC              *cloudformation.AWSEC2VPC
	Ig               *cloudformation.AWSEC2InternetGateway
	GWAttachment     *cloudformation.AWSEC2VPCGatewayAttachment
	PrivateACL       *cloudformation.AWSEC2NetworkAcl
	PublicACL        *cloudformation.AWSEC2NetworkAcl
	Ec2route         *cloudformation.AWSEC2Route
	PublicRouteTable *cloudformation.AWSEC2RouteTable
	HostedZone       *cloudformation.AWSRoute53HostedZone
	DhcpOpts         *cloudformation.AWSEC2DHCPOptions
	Sg               *cloudformation.AWSEC2SecurityGroup
}

func (v *VPC) commonSetupACL(t *goform.WrappedTemplate, acl *cloudformation.AWSEC2NetworkAcl) (*cloudformation.AWSEC2NetworkAclEntry, *cloudformation.AWSEC2NetworkAclEntry) {
	a := t.AttachResource(&cloudformation.AWSEC2NetworkAclEntry{
		CidrBlock: "0.0.0.0/0",
		Egress: false,
		NetworkAclId: t.MustRef(acl),
		Protocol: -1,
		RuleAction: "allow",
		RuleNumber: 100,
	}).(*cloudformation.AWSEC2NetworkAclEntry)
	b := t.AttachResource(&cloudformation.AWSEC2NetworkAclEntry{
		CidrBlock: "0.0.0.0/0",
		Egress: true,
		NetworkAclId: t.MustRef(acl),
		Protocol: -1,
		RuleAction: "allow",
		RuleNumber: 99,
	}).(*cloudformation.AWSEC2NetworkAclEntry)
	return a, b
}

func (v *VPC) setupDefaults() {
	if v.Config.VpcCidrBlock == "" {
		v.Config.VpcCidrBlock = "10.0.0.0/16"
	}
	if v.Config.SubnetBlocks == nil {
		v.Config.SubnetBlocks = []SubnetBlock {
			{
				PrivateBlock: "10.0.32.0/19",
				PublicBlock: "10.0.0.0/19",
			},
			{
				PrivateBlock: "10.0.96.0/19",
				PublicBlock: "10.0.64.0/19",
			},
			{
				PrivateBlock: "10.0.160.0/19",
				PublicBlock: "10.0.128.0/19",
			},
		}
	}
}

func (v *VPC) addOutputs(t *goform.WrappedTemplate, vpc *cloudformation.AWSEC2VPC, sg *cloudformation.AWSEC2SecurityGroup, subnets []*VPCSubnet) {
	private := make([]string, 0, len(subnets))
	public := make([]string, 0, len(subnets))
	for _, sub := range subnets {
		private = append(private, t.MustRef(sub.PrivateSubnet))
		public = append(public, t.MustRef(sub.PublicSubnet))
	}

	t.Outputs[v.Config.OutputPrefix + "PrivateSubnets"] = goform.Output{
		Description: "The private subnets of the VPC",
		Value: cloudformation.Join(",", private),
	}
	t.Outputs[v.Config.OutputPrefix + "PublicSubnets"] = goform.Output{
		Description: "The public subnets of the VPC",
		Value: cloudformation.Join(",", public),
	}
	t.Outputs[v.Config.OutputPrefix + "VPC"] = goform.Output{
		Description: "The VPC id",
		Value: t.MustRef(vpc),
	}
	t.Outputs[v.Config.OutputPrefix + "SecurityGroup"] = goform.Output{
		Description: "The default security groups of the VPC",
		Value: t.MustRef(sg),
	}
}

func (v *VPC) AddToTemplate(t *goform.WrappedTemplate) error {
	v.setupDefaults()
	v.VpcCidrBlock = t.AttachParameter("VipCidrBlock", &goform.Parameter{
		Default: stringPtr(v.Config.VpcCidrBlock),
		Description: "CidrBlock for the IP space for the the VPC",
		Type: "String",
	})
	v.VPC = t.AttachResource(&cloudformation.AWSEC2VPC{
		CidrBlock: t.MustRef(v.VpcCidrBlock),
		EnableDnsHostnames: true,
	}).(*cloudformation.AWSEC2VPC)
	v.Ig = t.AttachResource(&cloudformation.AWSEC2InternetGateway{}).(*cloudformation.AWSEC2InternetGateway)
	v.GWAttachment = t.AttachResource(&cloudformation.AWSEC2VPCGatewayAttachment{
		VpcId: t.MustRef(v.VPC),
		InternetGatewayId: t.MustRef(v.Ig),
	}).(*cloudformation.AWSEC2VPCGatewayAttachment)
	v.PrivateACL = t.AttachResource(&cloudformation.AWSEC2NetworkAcl{
		VpcId: t.MustRef(v.VPC),
	}).(*cloudformation.AWSEC2NetworkAcl)
	v.PublicACL = t.AttachResource(&cloudformation.AWSEC2NetworkAcl{
		VpcId: t.MustRef(v.VPC),
	}).(*cloudformation.AWSEC2NetworkAcl)
	v.commonSetupACL(t, v.PrivateACL)
	v.commonSetupACL(t, v.PublicACL)

	v.PublicRouteTable = t.AttachResource(&cloudformation.AWSEC2RouteTable{
		VpcId: t.MustRef(v.VPC),
	}).(*cloudformation.AWSEC2RouteTable)
	v.Ec2route = t.AttachResource(&cloudformation.AWSEC2Route{
		DestinationCidrBlock: "0.0.0.0/0",
		GatewayId: t.MustRef(v.Ig),
		RouteTableId: t.MustRef(v.PublicRouteTable),
	}).(*cloudformation.AWSEC2Route)
	v.HostedZone = t.AttachResource(&cloudformation.AWSRoute53HostedZone{
		HostedZoneConfig: &cloudformation.AWSRoute53HostedZone_HostedZoneConfig{
			Comment: "Interal service resolution for VPC resources",
		},
		Name: "internal.justin.tv",
		VPCs: []cloudformation.AWSRoute53HostedZone_VPC{
			{
				VPCId: t.MustRef(v.VPC),
				VPCRegion: cloudformation.Ref("AWS::Region"),
			},
		},
	}).(*cloudformation.AWSRoute53HostedZone)
	v.DhcpOpts = t.AttachResource(&cloudformation.AWSEC2DHCPOptions{
		DomainName: "us-west-2.compute.internal",
		DomainNameServers: []string {
			"AmazonProvidedDNS",
		},
	}).(*cloudformation.AWSEC2DHCPOptions)
	t.AttachResource(&cloudformation.AWSEC2VPCDHCPOptionsAssociation{
		DhcpOptionsId: t.MustRef(v.DhcpOpts),
		VpcId: t.MustRef(v.VPC),
	})
	v.Sg = t.AttachResource(&cloudformation.AWSEC2SecurityGroup{
		GroupName: "service_sg",
		GroupDescription: "service security group",
		SecurityGroupEgress: []cloudformation.AWSEC2SecurityGroup_Egress{
			{
				CidrIp: "0.0.0.0/0",
				IpProtocol: "-1",
			},
		},
		SecurityGroupIngress: []cloudformation.AWSEC2SecurityGroup_Ingress{
			{
				CidrIp: t.MustRef(v.VpcCidrBlock),
				IpProtocol: "-1",
			},
		},
		VpcId: t.MustRef(v.VPC),
	}).(*cloudformation.AWSEC2SecurityGroup)
	subnets := make([]*VPCSubnet, 0, len(v.Config.SubnetBlocks))
	for i := range v.Config.SubnetBlocks {
		vi := &VPCSubnet{
			SubnetConfig: SubnetConfig {
				VPCid: cloudformation.Ref(t.MustName(v.VPC)),
				PrivateACL: cloudformation.Ref(t.MustName(v.PrivateACL)),
				PublicACL: cloudformation.Ref(t.MustName(v.PublicACL)),
				PublicRouteTable: cloudformation.Ref(t.MustName(v.PublicRouteTable)),
				SubnetIndex: i,
				PublicSubnetBlock: v.Config.SubnetBlocks[i].PublicBlock,
				PrivateSubnetBlock: v.Config.SubnetBlocks[i].PrivateBlock,
			},
		}
		if err := vi.AddToTemplate(t); err != nil {
			return err
		}
		subnets = append(subnets, vi)
	}
	v.addOutputs(t, v.VPC, v.Sg, subnets)
	return nil
}