package simplevpn

import (
	"code.justin.tv/hygienic/goform"
	"github.com/awslabs/goformation/cloudformation"
	"strconv"
)

type SubnetConfig struct {
	VPCid string
	PrivateACL string
	PublicACL string
	PublicRouteTable string
	PublicSubnetBlock string
	PrivateSubnetBlock string
	// Which subnet in the list of subnets, usually 0 - 3
	SubnetIndex int
}

// VPCSubnet is one of the subnets per AZ for your VPN
type VPCSubnet struct {
	SubnetConfig SubnetConfig

	PrivateSubnet       *cloudformation.AWSEC2Subnet
	PublicSubnet        *cloudformation.AWSEC2Subnet
	EIP                 *cloudformation.AWSEC2EIP
	PrivateRouteTable   *cloudformation.AWSEC2RouteTable
	PublicSubnetBlock   *goform.Parameter
	PrivateSubnetBlock  *goform.Parameter
	Pubaclassoc         *cloudformation.AWSEC2SubnetNetworkAclAssociation
	Pubrtassoc          *cloudformation.AWSEC2SubnetRouteTableAssociation
	Privaclassoc        *cloudformation.AWSEC2SubnetNetworkAclAssociation
	Privrtassoc         *cloudformation.AWSEC2SubnetRouteTableAssociation
	Route               *cloudformation.AWSEC2Route
	NATFromPublicSubnet *cloudformation.AWSEC2NatGateway
}

func stringPtr(s string) *string {
	return &s
}

func (v *VPCSubnet) subnetSetupCommon(t *goform.WrappedTemplate, namePrefix string, MapPublicIpOnLaunch bool, routeTable string, subnetBlock *goform.Parameter) (*cloudformation.AWSEC2Subnet, *cloudformation.AWSEC2SubnetNetworkAclAssociation, *cloudformation.AWSEC2SubnetRouteTableAssociation) {
	subnet := t.AttachResource(&cloudformation.AWSEC2Subnet{
		AvailabilityZone: goform.SelectArr(strconv.Itoa(v.SubnetConfig.SubnetIndex), cloudformation.GetAZs("")),
		CidrBlock: t.MustRef(subnetBlock),
		MapPublicIpOnLaunch: true,
		VpcId: v.SubnetConfig.VPCid,
	}).(*cloudformation.AWSEC2Subnet)
	aclassoc := t.AttachResource(&cloudformation.AWSEC2SubnetNetworkAclAssociation{
		NetworkAclId: v.SubnetConfig.PublicACL,
		SubnetId: t.MustRef(subnet),
	}).(*cloudformation.AWSEC2SubnetNetworkAclAssociation)
	rtassoc := t.AttachResource(&cloudformation.AWSEC2SubnetRouteTableAssociation{
		RouteTableId: routeTable,
		SubnetId: t.MustRef(subnet),
	}).(*cloudformation.AWSEC2SubnetRouteTableAssociation)
	return subnet, aclassoc, rtassoc
}

func (v *VPCSubnet) AddToTemplate(t *goform.WrappedTemplate) error {
	suffix := string('A' + v.SubnetConfig.SubnetIndex)

	v.PublicSubnetBlock = t.AttachParameter("PublicSubnetBlock" + suffix, &goform.Parameter{
		Default: stringPtr(v.SubnetConfig.PublicSubnetBlock),
		Description: "Public CIDR block for subnet " + suffix,
		Type: "String",
	})
	v.PrivateSubnetBlock = t.AttachParameter("PrivateSubnetBlock" + suffix, &goform.Parameter{
		Default: stringPtr(v.SubnetConfig.PrivateSubnetBlock),
		Description: "Private CIDR block for subnet " + suffix,
		Type: "String",
	})
	v.EIP = t.AttachResource(&cloudformation.AWSEC2EIP{
		Domain: "vpc",
	}).(*cloudformation.AWSEC2EIP)
	v.PrivateRouteTable = t.AttachResource(&cloudformation.AWSEC2RouteTable{
		VpcId: v.SubnetConfig.VPCid,
	}).(*cloudformation.AWSEC2RouteTable)
	v.PublicSubnet, v.Pubaclassoc, v.Pubrtassoc = v.subnetSetupCommon(t, "public", true, v.SubnetConfig.PublicRouteTable, v.PublicSubnetBlock)
	v.PrivateSubnet, v.Privaclassoc, v.Privrtassoc =  v.subnetSetupCommon(t, "private", false, t.MustRef(v.PrivateRouteTable), v.PrivateSubnetBlock)

	v.NATFromPublicSubnet = t.AttachResource(&cloudformation.AWSEC2NatGateway{
		AllocationId: cloudformation.GetAtt(t.MustName(v.EIP), "AllocationId"),
		SubnetId: cloudformation.Ref(t.MustName(v.PublicSubnet)),
	}).(*cloudformation.AWSEC2NatGateway)
	v.Route = t.AttachResource(&cloudformation.AWSEC2Route{
		DestinationCidrBlock: "0.0.0.0/0",
		NatGatewayId:t.MustRef(v.NATFromPublicSubnet),
		RouteTableId: t.MustRef(v.PrivateRouteTable),
	}).(*cloudformation.AWSEC2Route)
	return nil
}