package spade

import (
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"
)

func TestConfig(t *testing.T) {
	Convey("StaticConfig", t, func() {
		Convey("when initialized", func() {
			config := StaticConfig{
				SpadeHost:           "https://www.arbitraryurl.com",
				BatchSize:           500,
				Timeout:             time.Minute,
				BacklogSize:         5000,
				Concurrency:         10,
				CombineSmallBatches: true,
			}

			Convey("GetSpadeHost should return its configured value", func() {
				So(config.GetSpadeHost(), ShouldEqual, config.SpadeHost)
			})

			Convey("GetBatchSize should return its configured value", func() {
				So(config.GetBatchSize(), ShouldEqual, config.BatchSize)
			})

			Convey("GetTimeout should return its configured value", func() {
				So(config.GetTimeout(), ShouldEqual, config.Timeout)
			})

			Convey("GetBacklogSize should return its configured value", func() {
				So(config.GetBacklogSize(), ShouldEqual, config.BacklogSize)
			})

			Convey("GetConcurrency should return its configured value", func() {
				So(config.GetConcurrency(), ShouldEqual, 10)
			})

			Convey("CombineSmallBatchesEnabled should return its configured value", func() {
				So(config.CombineSmallBatchesEnabled(), ShouldBeTrue)
			})
		})

		Convey("when uninitialized", func() {
			assertThatValuesAreReturned(t, &StaticConfig{})
		})
	})

	Convey("DynamicConfig", t, func() {
		Convey("when initialized", func() {
			config := DynamicConfig{
				SpadeHostFunc: func() string {
					return "https://www.arbitraryurl.com"
				},
				BatchSizeFunc: func() int64 {
					return 500
				},
				TimeoutFunc: func() time.Duration {
					return time.Minute
				},
				BacklogSizeFunc: func() int64 {
					return 5000
				},
				Concurrency: 10,
				CombineSmallBatchesFunc: func() bool {
					return true
				},
			}

			Convey("GetSpadeHost should return its configured value", func() {
				So(config.GetSpadeHost(), ShouldEqual, "https://www.arbitraryurl.com")
			})

			Convey("GetBatchSize should return its configured value", func() {
				So(config.GetBatchSize(), ShouldEqual, 500)
			})

			Convey("GetTimeout should return its configured value", func() {
				So(config.GetTimeout(), ShouldEqual, time.Minute)
			})

			Convey("GetBacklogSize should return its configured value", func() {
				So(config.GetBacklogSize(), ShouldEqual, 5000)
			})

			Convey("GetConcurrency should return its configured value", func() {
				So(config.GetConcurrency(), ShouldEqual, 10)
			})

			Convey("CombineSmallBatchesEnabled should return its configured value", func() {
				So(config.CombineSmallBatchesEnabled(), ShouldBeTrue)
			})
		})

		Convey("when uninitialized", func() {
			assertThatValuesAreReturned(t, &DynamicConfig{})
		})
	})
}

func assertThatValuesAreReturned(t *testing.T, config Config) {
	Convey("GetSpadeHost should return a non-zero value", func() {
		So(config.GetSpadeHost(), ShouldNotBeBlank)
	})

	Convey("GetBatchSize should return a non-zero value", func() {
		So(config.GetBatchSize(), ShouldBeGreaterThan, 0)
	})

	Convey("GetTimeout should return a non-zero value", func() {
		So(config.GetTimeout(), ShouldBeGreaterThan, 0)
	})

	Convey("GetBacklogSize should return a non-zero value", func() {
		So(config.GetBacklogSize(), ShouldBeGreaterThan, 0)
	})

	Convey("GetConcurrency should return a non-zero value", func() {
		So(config.GetConcurrency(), ShouldBeGreaterThan, 0)
	})

	Convey("CombineSmallBatchesEnabled should return false", func() {
		So(config.CombineSmallBatchesEnabled(), ShouldBeFalse)
	})
}
