// +build integration

package spade_test

import (
	"context"
	"encoding/json"
	"testing"
	"time"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/spade"
	"code.justin.tv/hygienic/spade/spadedconf"
	. "github.com/smartystreets/goconvey/convey"
)

type testEvent struct {
	Name string
}

type decodedStats struct {
	BacklogSize int64 `json:"backlog_size"`
	TotalEvents int64 `json:"total_events_sent"`
	AsyncErrors int64 `json:"async_errors"`
}

func TestIntegration(t *testing.T) {
	Convey("With setup", t, func() {
		dconf := &distconf.Distconf{}
		conf := &spadedconf.Config{}
		So(conf.Load(dconf), ShouldBeNil)
		c := spade.Client{
			Config: conf,
		}
		err := c.Setup()
		So(err, ShouldBeNil)
		ctx := context.Background()
		So(c.VerifyEndpoint(ctx), ShouldBeNil)

		event := spade.Event{
			Name: "testing_events",
			Properties: testEvent{
				Name: "testing-code",
			},
		}
		err = c.BlockingSendEvents(ctx, event)
		So(err, ShouldBeNil)

		// This is non blocking
		c.QueueEvents(event)

		var ds decodedStats
		So(json.Unmarshal([]byte(c.Vars().String()), &ds), ShouldBeNil)
		So(ds.BacklogSize, ShouldEqual, 1)
		So(ds.TotalEvents, ShouldEqual, 1)
		So(ds.AsyncErrors, ShouldEqual, 0)

		finished := make(chan struct{})
		go func() {
			_ = c.Start()
			close(finished)
		}()
		var i int
		for i = 0; i < 100; i++ {
			So(json.Unmarshal([]byte(c.Vars().String()), &ds), ShouldBeNil)
			if ds.TotalEvents == 2 {
				break
			}
			time.Sleep(time.Millisecond * 25)
		}

		// Second event was sent
		So(i, ShouldNotEqual, 100)
		So(ds.AsyncErrors, ShouldEqual, 0)
		So(c.Close(), ShouldBeNil)
		<-finished
	})
}
