package spade_test

import (
	"encoding/base64"
	"math/rand"
	"net/http"
	"net/url"
	"sync/atomic"
	"testing"
	"time"

	"code.justin.tv/hygienic/spade"
	"code.justin.tv/hygienic/spade/spadetest"
)

type clientTest struct {
	Config            *spade.StaticConfig
	HandlerDuration   time.Duration
	QueueInterval     time.Duration
	QueueEvents       int
	BatchesLowerBound int
	BatchesUpperBound int
}

func TestSpadeClient(t *testing.T) {
	tests := []clientTest{
		{
			Config: &spade.StaticConfig{
				BacklogSize: 100,
				BatchSize:   50,
			},
			QueueEvents:       100,
			BatchesLowerBound: 1,
			BatchesUpperBound: 10,
		},
		{
			Config: &spade.StaticConfig{
				BacklogSize: 100,
				BatchSize:   32,
				Concurrency: 10,
			},
			QueueEvents:       200,
			BatchesLowerBound: 30,
			BatchesUpperBound: 50,
		},
		{
			Config: &spade.StaticConfig{
				BacklogSize: 100,
				BatchSize:   10,
				Concurrency: 100,
			},
			QueueEvents:       1000,
			BatchesLowerBound: 900,
			BatchesUpperBound: 1000,
		},
		{
			Config: &spade.StaticConfig{
				BacklogSize:         100,
				BatchSize:           10,
				Concurrency:         100,
				CombineSmallBatches: true,
			},
			QueueEvents:       1000,
			BatchesLowerBound: 90,
			BatchesUpperBound: 110,
		},
		{
			Config: &spade.StaticConfig{
				BacklogSize:         100,
				BatchSize:           100,
				Concurrency:         100,
				CombineSmallBatches: true,
			},
			QueueEvents:       1000,
			BatchesLowerBound: 15,
			BatchesUpperBound: 30,
		},
	}
	for _, test := range tests {
		test := test

		t.Run("", func(t *testing.T) {
			if test.QueueInterval <= 0 {
				test.QueueInterval = time.Millisecond
			}
			if test.HandlerDuration <= 0 {
				test.HandlerDuration = 100 * time.Millisecond
			}

			// Setup fake spade server
			testServer := &spadetest.TestingSpadeServer{
				HandlerDuration: test.HandlerDuration,
			}
			go func() {
				if err := testServer.Start(); err != nil && err != http.ErrServerClosed {
					t.Logf("error starting spade test server: %v", err)
				}
			}()

			defer func() {
				err := testServer.Close()
				if err != nil {
					t.Logf("closing test server: %v", err)
				}
			}()

			conf := *test.Config
			conf.SpadeHost = testServer.ListenAddr()

			testReporter := &rememberingReporter{}
			testClient := &spade.Client{
				Config:   &conf,
				Reporter: testReporter,
			}

			if err := testClient.Setup(); err != nil {
				t.Errorf("unexpected Setup error: %q", err)
				return
			}

			go func() {
				if err := testClient.Start(); err != nil {
					t.Errorf("unexpected Start error: %q", err)
				}
			}()

			for i := 0; i < test.QueueEvents; i++ {
				testClient.QueueEvents(spade.Event{
					Name: "fake-event",
					Properties: map[string]interface{}{
						"test": true,
					},
				})
				time.Sleep(test.QueueInterval)
			}

			// Allow some time for the queue to drain before closing
			time.Sleep(2 * test.HandlerDuration)

			if err := testClient.Close(); err != nil {
				t.Errorf("unexpected Close error: %q", err)
				return
			}

			processed := int(testReporter.getEvents())
			if processed != test.QueueEvents {
				t.Errorf("expected %d but processed %d events", test.QueueEvents, testReporter.getEvents())
			}

			batches := int(testReporter.getRequestCount())
			if batches < test.BatchesLowerBound || batches > test.BatchesUpperBound {
				t.Errorf("expected [%d, %d] but processed %d batches",
					test.BatchesLowerBound, test.BatchesUpperBound, batches)
			}
		})
	}
}

type rememberingReporter struct {
	requestCount int64
	sendSuccess  int64
	sendFailure  int64
	events       int64
	backlogFull  int64
}

func (r *rememberingReporter) RequestCount() {
	atomic.AddInt64(&r.requestCount, 1)
}

func (r *rememberingReporter) SendSuccess(dur time.Duration) {
	atomic.AddInt64(&r.sendSuccess, 1)
}

func (r *rememberingReporter) SendFailure(dur time.Duration, err error) {
	atomic.AddInt64(&r.sendFailure, 1)
}

func (r *rememberingReporter) Events(eventCount int) {
	atomic.AddInt64(&r.events, int64(eventCount))
}

func (r *rememberingReporter) BacklogFull() {
	atomic.AddInt64(&r.backlogFull, 1)
}

func (r *rememberingReporter) getRequestCount() int64 {
	return atomic.LoadInt64(&r.requestCount)
}

func (r *rememberingReporter) getSendSuccess() int64 {
	return atomic.LoadInt64(&r.sendSuccess)
}

func (r *rememberingReporter) getSendFailure() int64 {
	return atomic.LoadInt64(&r.sendFailure)
}

func (r *rememberingReporter) getEvents() int64 {
	return atomic.LoadInt64(&r.events)
}

func (r *rememberingReporter) getBacklogFull() int64 {
	return atomic.LoadInt64(&r.backlogFull)
}

var letters = []rune("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ")

func generateLongStrings(length int) string {

	ret := make([]rune, length)
	for i := range ret {
		ret[i] = letters[rand.Intn(len(letters))]
	}
	return string(ret)
}

func TestBase64Padding(t *testing.T) {
	for i := 1; i < 100; i++ {
		src := generateLongStrings(i)

		// This is what the client used initially.
		// This is to ensure that the optimized approach yields the same results.
		expected := url.QueryEscape(base64.URLEncoding.EncodeToString([]byte(src)))

		got := string(spade.AddURLEscapedPadding([]byte(base64.RawURLEncoding.EncodeToString([]byte(src)))))

		if got != expected {
			t.Errorf("got: %s, expected: %s", got, expected)
		}

	}
}
