package taggingstatsmetrics

import (
	"time"

	"code.justin.tv/hygienic/spade"
)

type taggingSubStatter interface {
	IncD(key string, dimensions map[string]string, v int64)
	TimingDurationD(key string, dimensions map[string]string, v time.Duration)
}

// Reporter sends metrics to taggingSubStatter
type Reporter struct {
	// TaggingStatter sends statsd metrics
	TaggingStats taggingSubStatter
}

// RequestCount is called any time a spade request is made
func (r *Reporter) RequestCount() {
	r.inc("request_count", r.statTags("message"), 1)
}

// SendSuccess is called when a successful spade request happens. The HTTP response is an 204 (NoContent)
func (r *Reporter) SendSuccess(duration time.Duration) {
	r.timingDuration("send_timing", r.statTags("success"), duration)
}

// SendFailure is called when a spade request fails for any reason
func (r *Reporter) SendFailure(duration time.Duration, err error) {
	r.timingDuration("send_timing", r.statTags("fail"), duration)
}

// Events is called with the number of events sent in a successful request
func (r *Reporter) Events(eventCount int) {
	r.inc("events_count", r.statTags("events"), int64(eventCount))
}

// BacklogFull is called when the backlog is full and cannot queue more events
func (r *Reporter) BacklogFull() {
	r.inc("backlog_full", r.statTags("backlog"), 1)
}

func (r *Reporter) inc(stat string, dimensions map[string]string, value int64) {
	if r.TaggingStats == nil {
		return
	}

	r.TaggingStats.IncD(stat, dimensions, value)
}

func (r *Reporter) timingDuration(stat string, dimensions map[string]string, dur time.Duration) {
	if r.TaggingStats == nil {
		return
	}

	r.TaggingStats.TimingDurationD(stat, dimensions, dur)
}

func (r *Reporter) statTags(operationName string) map[string]string {
	return map[string]string{"origin": "spade", "Operation": operationName}
}

var _ spade.Reporter = (*Reporter)(nil)
