package spadetest

import (
	"encoding/base64"
	"encoding/json"
	"fmt"
	"io"
	"net"
	"net/http"
	"sync"
	"time"

	"context"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/spade"
)

// TestingSpadeServer allows testing spade events
type TestingSpadeServer struct {
	server   http.Server
	listener net.Listener

	Events          chan spade.Event
	HandlerDuration time.Duration

	listenErr error
	once      sync.Once
}

var _ http.Handler = &TestingSpadeServer{}

func (t *TestingSpadeServer) setup() {
	t.once.Do(func() {
		t.listener, t.listenErr = net.Listen("tcp", "0.0.0.0:0")
	})
}

// Drain the outstanding event queue
func (t *TestingSpadeServer) Drain() {
	for {
		select {
		case <-t.Events:
		default:
			return
		}
	}
}

// Find a matching event, waiting till ctx.  If the event is not found, return nil
func (t *TestingSpadeServer) Find(ctx context.Context, f func(e spade.Event) bool) *spade.Event {
	for {
		select {
		case e := <-t.Events:
			if f(e) {
				return &e
			}
		case <-ctx.Done():
			return nil
		}
	}
}

// SetupDistconf puts into distconf the current endpoint
func (t *TestingSpadeServer) SetupDistconf(mem distconf.Writer) error {
	return mem.Write("spade.endpoint", []byte(t.ListenAddr()))
}

// Start the testing spade server
func (t *TestingSpadeServer) Start() error {
	t.setup()
	if t.listenErr != nil {
		return t.listenErr
	}
	if t.Events == nil {
		t.Events = make(chan spade.Event, 1024)
	}
	t.server.Handler = t
	return t.server.Serve(t.listener)
}

// ListenAddr is the currently listening address of the test server
func (t *TestingSpadeServer) ListenAddr() string {
	t.setup()
	return fmt.Sprintf("http://127.0.0.1:%d", t.listener.Addr().(*net.TCPAddr).Port)
}

// Close ends the test server
func (t *TestingSpadeServer) Close() error {
	t.setup()
	return t.server.Close()
}

func (t *TestingSpadeServer) loopEvents(rw http.ResponseWriter, allEvents []spade.Event) {
	for _, e := range allEvents {
		select {
		case t.Events <- e:
		default:
			rw.WriteHeader(http.StatusInternalServerError)

		}
	}
}

// ServeHTTP pretends to be spade
func (t *TestingSpadeServer) ServeHTTP(rw http.ResponseWriter, req *http.Request) {
	if req.URL.Path == "/xarth" {
		if _, err := io.WriteString(rw, "XARTH"); err != nil {
			panic(err)
		}
		return
	}
	if req.URL.Path != "/track" {
		rw.WriteHeader(http.StatusNotFound)
		return
	}

	err := req.ParseForm()
	if err != nil {
		rw.WriteHeader(http.StatusBadRequest)
		return
	}

	var allEvents []spade.Event
	decodedBytes, err := base64.URLEncoding.DecodeString(req.Form.Get("data"))
	if err != nil {
		rw.WriteHeader(http.StatusBadRequest)
		if _, err2 := io.WriteString(rw, err.Error()); err2 != nil {
			panic(err2)
		}
		return
	}
	if err := json.Unmarshal(decodedBytes, &allEvents); err != nil {
		rw.WriteHeader(http.StatusBadRequest)
		if _, err2 := io.WriteString(rw, err.Error()); err2 != nil {
			panic(err2)
		}
		fmt.Println(req.URL.Query().Get("data"))
		return
	}
	t.loopEvents(rw, allEvents)
	if t.HandlerDuration > 0 {
		time.Sleep(t.HandlerDuration)
	}
	rw.WriteHeader(http.StatusNoContent)
}
