package sqsprocessor

import (
	"context"
	"errors"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
)

// MessageProcessor handles single SQS messages
type MessageProcessor interface {
	Process(ctx context.Context, msg *sqs.Message) error
}

// Config configures the processor
type Config struct {
	QueueURL          string
	VisibilityTimeout time.Duration
	WaitTime          time.Duration
	MsgReadDelay      time.Duration
	ListItemLimit     int
}

func (c *Config) fillDefaults() {
	if c.ListItemLimit == 0 {
		c.ListItemLimit = 1
	}
	if c.MsgReadDelay == 0 {
		c.MsgReadDelay = time.Second * 3
	}
	if c.WaitTime == 0 {
		c.WaitTime = time.Second * 3
	}
	if c.VisibilityTimeout == 0 {
		c.VisibilityTimeout = time.Minute
	}
}

// Circuit wraps io calls
type Circuit interface {
	Run(context.Context, func(context.Context) error) error
}

func callCircuit(ctx context.Context, in Circuit, f func(context.Context) error) error {
	if in == nil {
		return f(ctx)
	}
	return in.Run(ctx, f)
}

// QueueCircuits is the hystrix circuits for all the queue operations we care about
type QueueCircuits struct {
	Receive Circuit
	Delete  Circuit
	Send    Circuit
}

// Logger can log values
type Logger interface {
	Log(keyvals ...interface{})
}

// SQSProcessor makes it easy to read from an SQS queue
type SQSProcessor struct {
	SQS       *sqs.SQS
	Circuits  QueueCircuits
	Log       Logger
	Config    Config
	Processor MessageProcessor
	onDone    chan struct{}
}

// Setup should be called before Start
func (e *SQSProcessor) Setup() error {
	e.onDone = make(chan struct{})
	e.Config.fillDefaults()
	if e.Processor == nil {
		return errors.New("expect processor to be set")
	}
	return nil
}

// Start runs the processor.  Blocks.  Should call in goroutine
func (e *SQSProcessor) Start() error {
	var nextMsgDelay time.Duration
	for {
		select {
		case <-e.onDone:
			return nil
		case <-time.After(nextMsgDelay):
			nextMsgDelay = e.singleItem()
		}
	}
}

func (e *SQSProcessor) log(keyvals ...interface{}) {
	if e.Log != nil {
		e.Log.Log(keyvals...)
	}
}

// Close ends the Start call
func (e *SQSProcessor) Close() error {
	close(e.onDone)
	return nil
}

func (e *SQSProcessor) process(ctx context.Context, msg *sqs.Message) (err error) {
	defer func() {
		if r := recover(); r != nil {
			if err == nil {
				err = errors.New("panic recoverying message")
			}
			e.log("panic processing message")
		}
	}()
	return e.Processor.Process(ctx, msg)
}

func (e *SQSProcessor) singleItem() time.Duration {
	// SQS has a 12 hour limit on how long a msg can stay visible
	ctx, cancel := context.WithTimeout(context.Background(), time.Hour*12)
	defer cancel()
	msgs, err := e.receiveMessages(ctx)
	if err != nil {
		e.log("err", err, "unable to read new sqs message")
		return e.Config.MsgReadDelay
	}
	if len(msgs) == 0 {
		// This is when we should delay
		return e.Config.MsgReadDelay
	}
	sawFailure := false
	for _, msg := range msgs {
		if err := e.process(ctx, msg); err != nil {
			e.log("err", err, "unable to process SQS message")
			// This is when we should delay
			sawFailure = true
			continue
		}
		if err := e.deleteMessage(ctx, msg); err != nil {
			e.log("err", err, "unable to remove processed sqs message")
			// This is when we should delay
			sawFailure = true
		}
	}
	if sawFailure {
		return e.Config.MsgReadDelay
	}
	return time.Duration(0)
}

// SendMessage is a helper to add a message to the queue: Sets the QueueURL for you
func (e *SQSProcessor) SendMessage(ctx context.Context, in *sqs.SendMessageInput) error {
	in.QueueUrl = &e.Config.QueueURL
	req, _ := e.SQS.SendMessageRequest(in)
	return callCircuit(ctx, e.Circuits.Send, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
}

// ReceiveMessages reads and decodes messages from the queue
func (e *SQSProcessor) receiveMessages(ctx context.Context) ([]*sqs.Message, error) {
	req, out := e.SQS.ReceiveMessageRequest(&sqs.ReceiveMessageInput{
		QueueUrl:              aws.String(e.Config.QueueURL),
		AttributeNames:        []*string{aws.String("All")},
		MessageAttributeNames: []*string{aws.String("All")},
		VisibilityTimeout:     aws.Int64(int64(e.Config.VisibilityTimeout.Seconds()) + 1),
		WaitTimeSeconds:       aws.Int64(int64(e.Config.WaitTime.Seconds()) + 1),
	})
	err := callCircuit(ctx, e.Circuits.Receive, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	if err != nil {
		return nil, err
	}
	if len(out.Messages) == 0 {
		return nil, nil
	}
	return out.Messages, nil
}

func (e *SQSProcessor) deleteMessage(ctx context.Context, msg *sqs.Message) error {
	req, _ := e.SQS.DeleteMessageRequest(&sqs.DeleteMessageInput{
		QueueUrl:      aws.String(e.Config.QueueURL),
		ReceiptHandle: msg.ReceiptHandle,
	})
	err := callCircuit(ctx, e.Circuits.Delete, func(ctx context.Context) error {
		req.SetContext(ctx)
		return req.Send()
	})
	return err
}
