package sqsprotoprocessor

import (
	"bytes"
	"context"
	"encoding/base64"
	"io"

	"io/ioutil"
	"strings"

	"code.justin.tv/hygienic/sqsprocessor"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/golang/protobuf/proto"
)

const encodingKey = "X-SQS-Key"
const protoEncoding = "protobuf"

// Processor can process decoded Protobuf messages
type Processor interface {
	Process(ctx context.Context, msg *Message) error
}

// MessageSender can send protobuf messages: maybe you want to use sqsprocessor
type MessageSender interface {
	SendMessage(ctx context.Context, in *sqs.SendMessageInput) error
}

var _ MessageSender = &sqsprocessor.SQSProcessor{}

// ProtoProcessor is a SQS processor that helps encode/decode protobuf SQS messages
type ProtoProcessor struct {
	MessageFactory func() proto.Message
	Processor      Processor
}

// Message is a decoded SQS protobuf message
type Message struct {
	// ProtoMessage is the decoded protobuf message
	ProtoMessage proto.Message
	// SQSMessage is the original SQS message
	SQSMessage *sqs.Message
}

// Process decodes a SQS message that was encoded as protobuf serialized msg
func (p *ProtoProcessor) Process(ctx context.Context, msg *sqs.Message) error {
	protoMsg := p.MessageFactory()
	if err := decode(strings.NewReader(*msg.Body), protoMsg); err != nil {
		return err
	}
	return p.Processor.Process(ctx, &Message{
		ProtoMessage: protoMsg,
		SQSMessage:   msg,
	})
}

// encode the message we're going to send to SQS.  We have to URLEncoding the message because SQS is picky about
// any special characters
func encode(out io.Writer, in proto.Message) error {
	b, err := proto.Marshal(in)
	if err != nil {
		return err
	}
	c := base64.NewEncoder(base64.URLEncoding, out)
	_, err = io.Copy(c, bytes.NewReader(b))
	if err != nil {
		return err
	}
	return c.Close()
}

// decode a previously `encode` request
func decode(in io.Reader, out proto.Message) error {
	reader := base64.NewDecoder(base64.URLEncoding, in)
	b, err := ioutil.ReadAll(reader)
	if err != nil {
		return err
	}
	return proto.Unmarshal(b, out)
}

// ProtoMessageSender helps encode a SQS protobuf message
type ProtoMessageSender struct {
	MessageSender MessageSender
}

// SendMessage sends the encoded message to SQS
func (p *ProtoMessageSender) SendMessage(ctx context.Context, msg proto.Message, in *sqs.SendMessageInput) error {
	var buf bytes.Buffer
	if err := encode(&buf, msg); err != nil {
		return err
	}
	if in == nil {
		in = &sqs.SendMessageInput{}
	}
	bufBody := buf.String()
	in.MessageBody = &bufBody
	if in.MessageAttributes == nil {
		in.MessageAttributes = make(map[string]*sqs.MessageAttributeValue, 1)
	}
	encodingVal := protoEncoding
	dtype := "String"
	in.MessageAttributes[encodingKey] = &sqs.MessageAttributeValue{
		StringValue: &encodingVal,
		DataType:    &dtype,
	}
	return p.MessageSender.SendMessage(ctx, in)
}
