package statsdsender

import (
	"encoding/json"
	"sync/atomic"
	"time"
)

// The StatSender interface wraps all the statsd metric methods
type StatSender interface {
	Inc(string, int64, float32) error
	Dec(string, int64, float32) error
	Gauge(string, int64, float32) error
	GaugeDelta(string, int64, float32) error
	Timing(string, int64, float32) error
	TimingDuration(string, time.Duration, float32) error
	Set(string, string, float32) error
	SetInt(string, int64, float32) error
	Raw(string, string, float32) error
}

// ErrorTracker records errors that would otherwise be dropped on the floor
type ErrorTracker interface {
	Error(err error)
}

// AtomicErrorTracker increments an atomic value you can later expose on expvar
type AtomicErrorTracker struct {
	ErrorTracker ErrorTracker `nilcheck:"ignore"`
	count        int64
}

var _ ErrorTracker = &AtomicErrorTracker{}

// Error increments the atomic value and calls the sub tracker
func (a *AtomicErrorTracker) Error(err error) {
	if a != nil && err != nil {
		atomic.AddInt64(&a.count, 1)
		if a.ErrorTracker != nil {
			a.ErrorTracker.Error(err)
		}
	}
}

// MarshalJSON is a -race safe way to marshall this struct
func (a *AtomicErrorTracker) MarshalJSON() ([]byte, error) {
	return json.Marshal(struct {
		Count int64
	}{
		Count: atomic.LoadInt64(&a.count),
	})
}

// ErrorlessStatSender has non error returning functions that make it easier to use than a raw StatSender
type ErrorlessStatSender struct {
	StatSender
	ErrorTracker AtomicErrorTracker
}

func (st *ErrorlessStatSender) track(err error) {
	if err != nil {
		st.ErrorTracker.Error(err)
	}
}

// IncC is an error checked Inc
func (st *ErrorlessStatSender) IncC(s string, i int64, f float32) {
	st.track(st.StatSender.Inc(s, i, f))
}

// DecC is an error checked Dec
func (st *ErrorlessStatSender) DecC(s string, i int64, f float32) {
	st.track(st.StatSender.Dec(s, i, f))
}

// GaugeC is an error checked Gauge
func (st *ErrorlessStatSender) GaugeC(s string, i int64, f float32) {
	st.track(st.StatSender.Gauge(s, i, f))
}

// GaugeDeltaC is an error checked GaugeDelta
func (st *ErrorlessStatSender) GaugeDeltaC(s string, i int64, f float32) {
	st.track(st.StatSender.GaugeDelta(s, i, f))
}

// TimingC is an error checked Timing
func (st *ErrorlessStatSender) TimingC(s string, i int64, f float32) {
	st.track(st.StatSender.Timing(s, i, f))
}

// TimingDurationC is an error checked TimingDuration
func (st *ErrorlessStatSender) TimingDurationC(s string, t time.Duration, f float32) {
	st.track(st.StatSender.TimingDuration(s, t, f))
}

// SetC is an error checked Set
func (st *ErrorlessStatSender) SetC(s string, v string, f float32) {
	st.track(st.StatSender.Set(s, v, f))
}

// SetIntC is an error checked SetInt
func (st *ErrorlessStatSender) SetIntC(s string, i int64, f float32) {
	st.track(st.StatSender.SetInt(s, i, f))
}

// RawC is an error checked Raw
func (st *ErrorlessStatSender) RawC(s string, v string, f float32) {
	st.track(st.StatSender.Raw(s, v, f))
}
