package taggingcircuitsmetrics

import (
	"strings"
	"time"

	"sync"

	"github.com/cep21/circuit/v3"
	"github.com/cep21/circuit/v3/faststats"
	"github.com/cep21/circuit/v3/metrics/responsetimeslo"

	"code.justin.tv/hygienic/metricscactusstatsd"
)

// CommandFactory allows ingesting tagged metrics
type CommandFactory struct {
	StatSender metricscactusstatsd.TaggingSubStatter
	// This function will sanitize the circuit name and produce tags.  If you leave it empty, we will use a default implementation.
	// You are free to make this however you want to create tags.
	NameTagFunction func(name string) map[string]string
}

func (c *CommandFactory) nameTagFunction() func(name string) map[string]string {
	if c.NameTagFunction == nil {
		return nameTag
	}
	return c.NameTagFunction
}
func (c *CommandFactory) nameTag(name string) map[string]string {
	return c.nameTagFunction()(name)
}

func nameTag(s string) map[string]string {
	if len(s) > 64 {
		s = s[:64]
	}
	return map[string]string{"CircuitName": strings.Map(sanitizeRune, s)}
}

func sanitizeRune(r rune) rune {
	switch {
	case 'a' <= r && r <= 'z':
		return r
	case '0' <= r && r <= '9':
		return r
	case 'A' <= r && r <= 'Z':
		return r
	default:
		return '_'
	}
}

// ConcurrencyCollector runs as a background routine to collect concurrency stats
type ConcurrencyCollector struct {
	StatSender      metricscactusstatsd.TaggingSubStatter
	NameTagFunction func(name string) map[string]string
	Delay           faststats.AtomicInt64
	onClose         chan struct{}
	Manager         *circuit.Manager
	SampleRate      float32
	timeAfter       func(time.Duration) <-chan time.Time
	once            sync.Once
}

func (c *ConcurrencyCollector) delay() time.Duration {
	ret := c.Delay.Duration()
	if ret == 0 {
		return time.Second * 10
	}
	return ret
}

func (c *ConcurrencyCollector) after(dur time.Duration) <-chan time.Time {
	if c.timeAfter == nil {
		return time.After(dur)
	}
	return c.timeAfter(dur)
}

func (c *ConcurrencyCollector) init() {
	c.once.Do(func() {
		c.onClose = make(chan struct{})
	})
}

// Collect reports concurrency information for each circuit
func (c *ConcurrencyCollector) Collect() {
	for _, circ := range c.Manager.AllCircuits() {
		name := circ.Name()
		concurrent := circ.ConcurrentCommands()
		maxConcurrent := circ.Config().Execution.MaxConcurrentRequests

		p := c.StatSender.NewDimensionalSubStatter(combineDimensions(map[string]string{"Producer": "circuit"}, c.NameTagFunction(name)))

		isOpen := int64(0)
		if circ.IsOpen() {
			isOpen = 1
		}
		p.GaugeD("is_open", nil, isOpen)
		p.GaugeD("concurrent", nil, concurrent)
		p.GaugeD("max_concurrent", nil, maxConcurrent)
	}
}

// Start blocks forever and runs circuit collection on a delay of variable `Delay`
func (c *ConcurrencyCollector) Start() {
	c.init()
	for {
		select {
		case <-c.onClose:
			return
		case <-c.after(c.delay()):
			c.Collect()
		}
	}
}

// Close stops the `Start` routine
func (c *ConcurrencyCollector) Close() error {
	c.init()
	close(c.onClose)
	return nil
}

// ConcurrencyCollector returns a ConcurrencyCollector, which can be used to collect stats on each circuit tracked
// by the manager.  You should call `Start` on the returned object and `Close` when you are done collecting metrics.
func (c *CommandFactory) ConcurrencyCollector(manager *circuit.Manager) *ConcurrencyCollector {
	return &ConcurrencyCollector{
		Manager:         manager,
		StatSender:      c.StatSender,
		NameTagFunction: c.nameTagFunction(),
	}
}

// SLOCollector tracks SLO stats for statsd
func (c *CommandFactory) SLOCollector(circuitName string) responsetimeslo.Collector {
	return &SLOCollector{
		TaggingSubStatter: c.StatSender.NewDimensionalSubStatter(
			combineDimensions(map[string]string{"Producer": "circuit.slo"},
				c.nameTag(circuitName))),
	}
}

// CommandProperties creates statsd metrics for a circuit
func (c *CommandFactory) CommandProperties(circuitName string) circuit.Config {
	return circuit.Config{
		Metrics: circuit.MetricsCollectors{
			Run: []circuit.RunMetrics{
				&RunMetricsCollector{
					TaggingSubStatter: c.StatSender.NewDimensionalSubStatter(combineDimensions(map[string]string{"Producer": "circuit.run"}, c.nameTag(circuitName))),
				},
			},
			Fallback: []circuit.FallbackMetrics{
				&FallbackMetricsCollector{
					TaggingSubStatter: c.StatSender.NewDimensionalSubStatter(combineDimensions(map[string]string{"Producer": "circuit.fallback"}, c.nameTag(circuitName))),
				},
			},
			Circuit: []circuit.Metrics{
				&CircuitMetricsCollector{
					TaggingSubStatter: c.StatSender.NewDimensionalSubStatter(combineDimensions(map[string]string{"Producer": "circuit"}, c.nameTag(circuitName))),
				},
			},
		},
	}
}

func combineDimensions(dims ...map[string]string) map[string]string {
	newDims := make(map[string]string)
	for _, dim := range dims {
		for k, v := range dim {
			newDims[k] = v
		}
	}
	return newDims
}

// CircuitMetricsCollector collects opened/closed metrics
type CircuitMetricsCollector struct {
	metricscactusstatsd.TaggingSubStatter
	SampleRate float32
}

// Closed sets a gauge as closed for the collector
func (c *CircuitMetricsCollector) Closed(now time.Time) {
	c.GaugeD("is_open", nil, 0)
}

// Opened sets a gauge as opened for the collector
func (c *CircuitMetricsCollector) Opened(now time.Time) {
	c.GaugeD("is_open", nil, 1)
}

var _ circuit.Metrics = &CircuitMetricsCollector{}

// SLOCollector collects SLO level metrics
type SLOCollector struct {
	metricscactusstatsd.TaggingSubStatter
	SampleRate float32
}

// Failed increments a failed metric
func (s *SLOCollector) Failed() {
	s.IncD("failed", nil, 1)
}

// Passed increments a passed metric
func (s *SLOCollector) Passed() {
	s.IncD("passed", nil, 1)
}

var _ responsetimeslo.Collector = &SLOCollector{}

// RunMetricsCollector collects command metrics
type RunMetricsCollector struct {
	metricscactusstatsd.TaggingSubStatter
	SampleRate float32
}

// Success sends a success to statsd
func (c *RunMetricsCollector) Success(now time.Time, duration time.Duration) {
	c.IncD("success", nil, 1)
	c.TimingDurationD("calls", nil, duration)
}

// ErrFailure sends a failure to statsd
func (c *RunMetricsCollector) ErrFailure(now time.Time, duration time.Duration) {
	c.IncD("err_failure", nil, 1)
	c.TimingDurationD("calls", nil, duration)
}

// ErrTimeout sends a timeout to statsd
func (c *RunMetricsCollector) ErrTimeout(now time.Time, duration time.Duration) {
	c.IncD("err_timeout", nil, 1)
	c.TimingDurationD("calls", nil, duration)
}

// ErrBadRequest sends a bad request error to statsd
func (c *RunMetricsCollector) ErrBadRequest(now time.Time, duration time.Duration) {
	c.IncD("err_bad_request", nil, 1)
	c.TimingDurationD("calls", nil, duration)
}

// ErrInterrupt sends an interrupt error to statsd
func (c *RunMetricsCollector) ErrInterrupt(now time.Time, duration time.Duration) {
	c.IncD("err_interrupt", nil, 1)
	c.TimingDurationD("calls", nil, duration)
}

// ErrShortCircuit sends a short circuit to statsd
func (c *RunMetricsCollector) ErrShortCircuit(now time.Time) {
	c.IncD("err_short_circuit", nil, 1)
}

// ErrConcurrencyLimitReject sends a concurrency limit error to statsd
func (c *RunMetricsCollector) ErrConcurrencyLimitReject(now time.Time) {
	c.IncD("err_concurrency_limit_reject", nil, 1)
}

var _ circuit.RunMetrics = &RunMetricsCollector{}

// FallbackMetricsCollector collects fallback metrics
type FallbackMetricsCollector struct {
	metricscactusstatsd.TaggingSubStatter
	SampleRate float32
}

// Success sends a success to statsd
func (c *FallbackMetricsCollector) Success(now time.Time, duration time.Duration) {
	c.IncD("success", nil, 1)
}

// ErrConcurrencyLimitReject sends a concurrency-limit to statsd
func (c *FallbackMetricsCollector) ErrConcurrencyLimitReject(now time.Time) {
	c.IncD("err_concurrency_limit_reject", nil, 1)
}

// ErrFailure sends a failure to statsd
func (c *FallbackMetricsCollector) ErrFailure(now time.Time, duration time.Duration) {
	c.IncD("err_failure", nil, 1)
}

var _ circuit.FallbackMetrics = &FallbackMetricsCollector{}
