package twirpclienthooks

import (
	"net/http"

	"github.com/twitchtv/twirp"
	"github.com/twitchtv/twirp/ctxsetters"
)

// RoundTripper implements the http.RoundTripper interface and reports statsd client side twirp metrics.
type RoundTripper struct {
	Transport http.RoundTripper
	Hooks     *twirp.ServerHooks
}

var _ http.RoundTripper = &RoundTripper{}

// RoundTrip executes the http.Request and reprots client side statsd metrics via twirp Hooks.
func (mrt *RoundTripper) RoundTrip(req *http.Request) (*http.Response, error) {
	if mrt == nil {
		return http.DefaultTransport.RoundTrip(req)
	}

	transport := mrt.getTransport()
	if mrt.Hooks == nil {
		return transport.RoundTrip(req)
	}

	ctx := req.Context()

	if mrt.Hooks.RequestReceived != nil {
		var err error
		ctx, err = mrt.Hooks.RequestReceived(ctx)
		if err != nil {
			return transport.RoundTrip(req)
		}
	}

	if mrt.Hooks.RequestRouted != nil {
		var err error
		ctx, err = mrt.Hooks.RequestRouted(ctx)
		if err != nil {
			return transport.RoundTrip(req)
		}
	}

	req = req.WithContext(ctx)
	res, err := transport.RoundTrip(req)

	ctx = ctxsetters.WithStatusCode(ctx, getStatusCode(res)) // required to call ResponseSent hook
	if mrt.Hooks.ResponseSent != nil {
		mrt.Hooks.ResponseSent(ctx)
	}
	if mrt.Hooks.Error != nil && err != nil {
		// No use for the context returned by Error at this point: it is ignored
		mrt.Hooks.Error(ctx, twirp.InternalErrorWith(err))
	}
	return res, err
}

// getTransport returns the inner http.RoundTripper.
// getTransport returns http.DefaultTransport if RoundTripper.Transport is nil.
func (mrt *RoundTripper) getTransport() http.RoundTripper {
	if mrt.Transport == nil {
		return http.DefaultTransport
	}
	return mrt.Transport
}

func getStatusCode(res *http.Response) int {
	if res == nil {
		return 0
	}
	return res.StatusCode
}
