// Generates an SLO map file from godoc
package main

import (
	"bytes"
	"errors"
	"fmt"
	"go/ast"
	"go/format"
	"go/parser"
	"go/token"
	"html/template"
	"io/ioutil"
	"os"
	"regexp"
	"strings"
	"time"

	"code.justin.tv/hygienic/twirpserviceslohook"
)

type generator struct {
	sourceFile  string
	args        []string
	packageName string
}

func main() {
	g := generator{
		args: os.Args[1:],
	}
	if err := g.main(); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}

func (g *generator) setupFlags() error {
	if len(g.args) != 2 {
		return errors.New("usage: slogenerator <source_twirp_go_rpc_file> <package_name>")
	}
	g.sourceFile = g.args[0]
	g.packageName = g.args[1]
	return nil
}

func (g *generator) nodeParser(f ast.Node) map[twirpserviceslohook.ServiceName]map[twirpserviceslohook.MethodName]time.Duration {
	reg := regexp.MustCompile(`SLO@(\S*)`)
	SLO := map[twirpserviceslohook.ServiceName]map[twirpserviceslohook.MethodName]time.Duration{}
	ast.Inspect(f, func(n ast.Node) bool {
		switch t := n.(type) {
		case *ast.TypeSpec:
			// which are public
			if t.Name.IsExported() {
				exprType, ok := t.Type.(*ast.InterfaceType)
				if ok {
					if SLO[twirpserviceslohook.ServiceName(t.Name.Name)] == nil {
						SLO[twirpserviceslohook.ServiceName(t.Name.Name)] = map[twirpserviceslohook.MethodName]time.Duration{}
					}
					for _, m := range exprType.Methods.List {
						docText := m.Doc.Text()
						match := reg.FindStringSubmatch(docText)
						if match != nil {
							matchAsTime, err := time.ParseDuration(match[1])
							if err == nil {
								SLO[twirpserviceslohook.ServiceName(t.Name.Name)][twirpserviceslohook.MethodName(m.Names[0].Name)] = matchAsTime
							}
						}
					}
					if len(SLO[twirpserviceslohook.ServiceName(t.Name.Name)]) == 0 {
						delete(SLO, twirpserviceslohook.ServiceName(t.Name.Name))
					}
				}
			}
		}
		return true
	})
	return SLO
}

func (g *generator) main() error {
	if err := g.setupFlags(); err != nil {
		return err
	}
	readFile, err := ioutil.ReadFile(g.sourceFile)
	if err != nil {
		return fmt.Errorf("unable to read source file %s: %s", g.sourceFile, err)
	}
	fset := token.NewFileSet()
	f, err := parser.ParseFile(fset, g.sourceFile, readFile, parser.ParseComments)
	if err != nil {
		return fmt.Errorf("unable to parse source file %s: %s", g.sourceFile, err)
	}
	SLO := g.nodeParser(f)

	templateOut := struct {
		Package string
		SLO     map[twirpserviceslohook.ServiceName]map[twirpserviceslohook.MethodName]time.Duration
	}{
		Package: g.packageName,
		SLO:     SLO,
	}
	gofileTemplate := `// Code generated by slogenerator. DO NOT EDIT.
package {{ .Package }}

import (
	"time"

	"code.justin.tv/hygienic/twirpserviceslohook"
)

var _ twirpserviceslohook.ServiceName = ""
var _ time.Duration = time.Second

{{ range $key, $value := .SLO -}}
// SLO{{ $key }} is auto generated and helps track method call SLO
var SLO{{ $key }} = map[twirpserviceslohook.ServiceName]map[twirpserviceslohook.MethodName]time.Duration{
	twirpserviceslohook.ServiceName("{{ $key }}"): {
		{{- range $key2, $value2 := $value }}
		twirpserviceslohook.MethodName("{{ $key2 }}"): {{ $value2.Nanoseconds }},
		{{- end }}
	},
}
{{ end -}}`
	t := template.Must(template.New("").Parse(gofileTemplate))
	var buf bytes.Buffer
	if err2 := t.Execute(&buf, templateOut); err2 != nil {
		return fmt.Errorf("unable to execute template: %s", err2.Error())
	}
	fileSource := strings.TrimSpace(buf.String())
	formattedBytes, err3 := format.Source([]byte(fileSource))
	if err3 != nil {
		fmt.Println(fileSource)
	} else {
		fmt.Println(strings.TrimSpace(string(formattedBytes)))
	}
	return nil
}
