package taggingslo

import (
	"strings"

	"code.justin.tv/hygienic/twirpserviceslohook"
)

// StatTracker implements the stat tracker interface for twirpserviceslohook, but sends metrics to a tagging based metric service
type StatTracker struct {
	// How to sanitize method and service names.  By default, allows [a-zA-Z0-9_]*
	Sanitizer func(string) string `json:"-"`
	// Statter is something that can increment statsd metrics. Matches the normal Go statsd interface
	Statter Statter `json:"-"`
}

// Statter is similar to the statsd inc interface
type Statter interface {
	IncD(key string, dimensions map[string]string, v int64)
}

func sanitize(s string) string {
	return strings.Map(sanitizeRune, s)
}

func sanitizeRune(r rune) rune {
	switch {
	case 'a' <= r && r <= 'z':
		return r
	case '0' <= r && r <= '9':
		return r
	case 'A' <= r && r <= 'Z':
		return r
	default:
		return '_'
	}
}

var _ twirpserviceslohook.StatTracker = &StatTracker{}

func (s *StatTracker) sanitize(in string) string {
	if s.Sanitizer == nil {
		return sanitize(in)
	}
	return s.Sanitizer(in)
}

func (s *StatTracker) common(serviceName string, methodName string, suffix string) {
	statDimensions := map[string]string{"service": s.sanitize(serviceName), "method": s.sanitize(methodName)}
	s.Statter.IncD(suffix, statDimensions, 1)
}

// Passed will increment a 'passed' metric to the Statter
func (s *StatTracker) Passed(serviceName string, methodName string) {
	s.common(serviceName, methodName, "passed")
}

// Failed will increment a 'failed' metric to the Statter
func (s *StatTracker) Failed(serviceName string, methodName string) {
	s.common(serviceName, methodName, "failed")
}
