package telemetryslo

import (
	"strings"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/hygienic/twirpserviceslohook"
)

// StatTracker implements the stat tracker interface for twirpserviceslohook, and sends metrics to twitch telemetry
//
// Metrics reported. The Operation dimension is set to the Twirp method:
//
// * SLOPassed - Sum or Average statistic is most useful. This is always recorded as 1 or 0
// * SLOFailed - Sum or Average statistic is most useful. This is always recorded as 1 or 0
type StatTracker struct {
	// How to sanitize method and service names.  By default, allows [a-zA-Z0-9_]*
	Sanitizer func(string) string `json:"-"`
	// SampleReporter reports metrics
	SampleReporter *telemetry.SampleReporter `json:"-"`
}

func defaultSanitize(s string) string {
	return strings.Map(sanitizeRune, s)
}

func sanitizeRune(r rune) rune {
	switch {
	case 'a' <= r && r <= 'z':
		return r
	case '0' <= r && r <= '9':
		return r
	case 'A' <= r && r <= 'Z':
		return r
	default:
		return '_'
	}
}

var _ twirpserviceslohook.StatTracker = &StatTracker{}

func (s *StatTracker) sanitize(in string) string {
	if s.Sanitizer == nil {
		return defaultSanitize(in)
	}
	return s.Sanitizer(in)
}

// Passed will increment a SLOPassed metric by 1 and SLOFailed metric by 0
func (s *StatTracker) Passed(_ string, methodName string) {
	s.reportPassed(methodName, true)
}

// Failed will increment a SLOPassed metric by 0 and SLOFailed metric by 1
func (s *StatTracker) Failed(_ string, methodName string) {
	s.reportPassed(methodName, false)
}

func (s *StatTracker) reportPassed(methodName string, passed bool) {
	if s.SampleReporter == nil {
		return
	}

	// Create a copy of struct (convention to change dimensions)
	var reporter telemetry.SampleReporter
	reporter = *s.SampleReporter
	reporter.OperationName = s.sanitize(methodName)

	// Always record "SLOPassed" or "SLOFailed". This allows using 'Average" as a statistic to graph percentage
	if passed {
		reporter.Report("SLOPassed", 1, telemetry.UnitCount)
		reporter.Report("SLOFailed", 0, telemetry.UnitCount)
	} else {
		reporter.Report("SLOPassed", 0, telemetry.UnitCount)
		reporter.Report("SLOFailed", 1, telemetry.UnitCount)
	}
}
