package telemetryslo

import (
	"reflect"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
)

const samplesPerReport = 2

func TestStatTracker(t *testing.T) {
	observer := &rememberingSampleObserver{}

	const serviceName = "TestService"

	reporter := telemetry.SampleReporter{
		SampleBuilder: telemetry.SampleBuilder{
			ProcessIdentifier: identifier.ProcessIdentifier{
				Service: serviceName,
			},
		},
		SampleObserver: observer,
	}

	s := StatTracker{
		SampleReporter: &reporter,
	}

	s.Passed(serviceName, "OperationA")
	s.Failed(serviceName, "OperationB")
	require.Len(t, observer.samples, 2*samplesPerReport)

	assertContainsSample(t, observer.samples, "SLOPassed", map[string]string{
		"Service":   serviceName,
		"Operation": "OperationA",
	}, 1)
	assertContainsSample(t, observer.samples, "SLOFailed", map[string]string{
		"Service":   serviceName,
		"Operation": "OperationA",
	}, 0)

	assertContainsSample(t, observer.samples, "SLOPassed", map[string]string{
		"Service":   serviceName,
		"Operation": "OperationB",
	}, 0)
	assertContainsSample(t, observer.samples, "SLOFailed", map[string]string{
		"Service":   serviceName,
		"Operation": "OperationB",
	}, 1)
}

func assertContainsSample(t *testing.T, samples []*telemetry.Sample, metricName string, dimensions telemetry.DimensionSet, value float64) {
	for _, sample := range samples {
		if sample.MetricID.Name == metricName && reflect.DeepEqual(sample.MetricID.Dimensions, dimensions) && sample.Value == value {
			return
		}
	}
	assert.Failf(t, "samples is missing expected sample", "metricName=%v dimensions=%v value=%v", metricName, dimensions, value)
}

type rememberingSampleObserver struct {
	samples []*telemetry.Sample
}

func (r *rememberingSampleObserver) ObserveSample(sample *telemetry.Sample) {
	r.samples = append(r.samples, sample)
}

func (r *rememberingSampleObserver) Flush() {}
func (r *rememberingSampleObserver) Stop()  {}
