package internal

import (
	"context"
	"expvar"
	"net"
	"net/http"
	"time"

	"code.justin.tv/hygienic/awsexpvar"
	"code.justin.tv/hygienic/dialersocks/socks"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/expvar2"
	"code.justin.tv/hygienic/rlimit"
	"code.justin.tv/hygienic/twitchbaseservice/v4"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
)

// VerifyDialContext checks if an instance metadata request succeeds using DialContext()
func VerifyDialContext(s twitchbaseservice.BaseService) error {
	client := http.Client{
		Transport: &http.Transport{
			DialContext: s.DialContext(),
		},
	}
	// Verify the socks proxy to an AWS host
	req, err := http.NewRequest(http.MethodGet, "http://169.254.169.254/latest/meta-data/", nil)
	if err != nil {
		return err
	}
	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	req = req.WithContext(ctx)
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	defer func() {
		if closeErr := resp.Body.Close(); closeErr != nil {
			s.Logger().Log("err", closeErr, "closing body")
		}
	}()
	if resp.StatusCode != http.StatusOK {
		return errors.Errorf("unknown status code %d", resp.StatusCode)
	}
	return nil
}

func VerifyAWSConn(s twitchbaseservice.BaseService) error {
	awsSession, err := session.NewSession(s.AWSConfig())
	if err != nil {
		return errors.Wrap(err, "unable to create aws session")
	}
	stsClient := sts.New(awsSession)
	identOut, err := stsClient.GetCallerIdentity(nil)
	if err != nil {
		return errors.Wrap(err, "unable to use AWS credentials to get caller identity")
	}
	s.Logger().Log("aws_account", *identOut.Account, "userID", *identOut.UserId, "arn", *identOut.Arn, "aws session verified")
	return nil
}

func CreateDialContext(c twitchbaseservice.BaseService, env string, config twitchbaseservice.OptionalConfig) func(ctx context.Context, network, addr string) (net.Conn, error) {
	if env == twitchbaseservice.DevEnvironment {
		d := socks.NewDialer("tcp", SocksAddr(c))
		if config.RootDialer != nil {
			d.ProxyDial = config.RootDialer
		}
		return d.DialContext
	}
	if config.RootDialer == nil {
		var d net.Dialer
		return d.DialContext
	}
	return config.RootDialer
}

func SetupExpvar(s twitchbaseservice.BaseWithExpvar) {
	startTime := time.Now()

	s.ExpvarHandler().Exported["env"] = expvar2.EnviromentalVariables()
	s.ExpvarHandler().Exported["conf"] = s.Parameters().Var()
	s.ExpvarHandler().Exported["circuits"] = s.CircuitManager().Var()
	s.ExpvarHandler().Exported["uptime"] = expvar.Func(func() interface{} {
		return time.Since(startTime).String()
	})
	s.ExpvarHandler().Exported["secrets"] = expvar.Func(func() interface{} {
		return s.Secrets().Keys()
	})
	var awsExpvarClient awsexpvar.Expvar
	s.ExpvarHandler().Exported["aws-metadata"] = awsExpvarClient.Var()
	s.ExpvarHandler().Exported["rlimits"] = rlimit.Var()
}

func SocksAddr(s twitchbaseservice.BaseService) string {
	return s.Parameters().Str("socks_addr", "localhost:4456").Get()
}
