package twitchbaseservice

import (
	"context"
	"net"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/expvar2"
	"code.justin.tv/hygienic/log"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/cep21/circuit"
	"github.com/twitchtv/twirp"
)

// DevEnvironment is a special const that means you're running locally from a mac and need a SOCKS proxy
const DevEnvironment = "development"

// OptionalConfig is config that is optional
type OptionalConfig struct {
	DevelopmentProfile  string
	FallbackEnvironment *string

	Environment string
	Region      string
	SocksAddr   string
	RootDialer  func(ctx context.Context, network, addr string) (net.Conn, error)
	// If true, the debug service is not setup
	IgnoreDebugService       bool
	DisableSocksVerification bool
	OsGetEnv                 func(string) string
	ForcedLogger             log.Logger
	DefaultReaders           []distconf.Reader
}

// BaseService gives you a common framework of dependent libraries your service can trust to have.  Implementations
// are responsible for creating this setup.
type BaseService interface {
	// parameters configure non secret parts of your service
	Parameters() *distconf.Distconf
	// Secrets are for secret information that must be protected, like API keys
	Secrets() *distconf.Distconf
	// Logger is how you should log
	Logger() log.Logger
	// CircuitManager is for outbound requests
	CircuitManager() *circuit.Manager
	// Statsd is for basemetrics
	Statsd() statsd.SubStatter
	// DialContext describes how to dial network addresses.  Use this, instead of Go's built in dialer, to support
	// local execution via SOCKS proxies.
	DialContext() func(ctx context.Context, network, addr string) (net.Conn, error)
	// This aws config should be used for all AWS calls
	AWSConfig() *aws.Config
	// Runner should be used to start and eventually stop both your service, and anything the base service uses
	Close() error
}

// WithExpvar is an optional stack extension that supports expvar variables
type WithExpvar interface {
	// This expvar handler should eventually be exposed so we can profile information
	ExpvarHandler() *expvar2.Handler
}

type BaseWithExpvar interface {
	BaseService
	WithExpvar
}

// WithTwirpHook is an optional stack extension that provides an appropriate basic monitoring twirp hook
type WithTwirpHooks interface {
	// the hook returned from this should be used to simplify eventual changes to the underling metrics service
	TwirpHooks() *twirp.ServerHooks
}
