package twitchbasemetriccloudwatch

import (
	"sync"

	"code.justin.tv/amzn/TwitchProcessIdentifier"
	"code.justin.tv/hygienic/cloudwatchmetrics"
	"code.justin.tv/hygienic/cwmessagebatch"
	"code.justin.tv/hygienic/metrics"
	"code.justin.tv/hygienic/metrics/metricsext"
	"code.justin.tv/hygienic/seh1"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
)

type Logger interface {
	Log(keyvals ...interface{})
}

// Copy/paste from https://git-aws.internal.justin.tv/amzn/TwitchProcessIdentifier

type Service struct {
	// Base must have a valid aws config and valid logger before you call this service
	ProcessIdentifier identifier.ProcessIdentifier
	Config            *aws.Config
	Logger            Logger
	ExtraSinks        []metrics.AggregationSink
	ExtraSources      []metrics.AggregationSource

	// on struct so it can be returned by Registry()
	baseRegistry metrics.BaseRegistry
	// On struct so we can start/stop these services
	bufferedSink metricsext.BufferedSink
	flusher      metricsext.PeriodicFlusher
	setupOnce    bool
}

func (m *Service) Registry() metrics.BaseRegistry {
	return m.baseRegistry
}

func (m *Service) log(keyvals ...interface{}) {
	if m.Logger != nil {
		m.Logger.Log(keyvals...)
	}
}

func (m *Service) Setup() error {
	if m.setupOnce {
		return nil
	}
	m.setupOnce = true
	ses, err := session.NewSession(m.Config)
	if err != nil {
		return err
	}
	ca := cloudwatchmetrics.CloudwatchAggregator{
		Config: cloudwatchmetrics.Config{
			Namespace: m.ProcessIdentifier.Service,
		},
		Sender: cwmessagebatch.Aggregator{
			Client: cloudwatch.New(ses),
			Config: cwmessagebatch.Config{
				OnDroppedDatum: func(datum *cloudwatch.MetricDatum) {
					m.log("dropped data on send to cloudwatch")
				},
			},
		},
	}
	rootRegistry := &metrics.Registry{
		AggregationConstructor: seh1.SEHRollingAggregation,
	}
	m.baseRegistry = rootRegistry
	m.bufferedSink = metricsext.BufferedSink{
		Destination: &ca,
		Config: metricsext.BufferConfig{
			OnDroppedAggregation: func(e error, aggregations []metrics.TimeSeriesAggregation) {
				m.log("err", e, "len", len(aggregations), "Aggregation dropped in buffer")
			},
		},
	}
	dims := dims(m.ProcessIdentifier)
	if len(dims) > 0 {
		m.baseRegistry = metricsext.WithDimensions(m.baseRegistry, dims)
		if dims["Substage"] != "" {
			m.baseRegistry = metricsext.RegistryWithRollup(m.baseRegistry, []string{"Substage"})
		}
	}
	sink := metricsext.MultiSink{}
	source := metricsext.MultiSource{}

	source.AddSource(rootRegistry)
	for _, s := range m.ExtraSources {
		source.AddSource(s)
	}
	sink.AddSink(&m.bufferedSink)
	for _, s := range m.ExtraSinks {
		sink.AddSink(s)
	}

	rollupSink := metricsext.RollupSink{
		Sink: &metricsext.CompressionSink{
			Sink: &sink,
		},
		TSSource: rootRegistry,
	}
	m.flusher = metricsext.PeriodicFlusher{
		Flushable: &metricsext.AggregationFlusher{
			Source: &source,
			Sink:   &rollupSink,
		},
	}
	return nil
}

func (m *Service) Start() error {
	wg := sync.WaitGroup{}
	wg.Add(2)
	var errs [2]error
	go func() {
		defer wg.Done()
		errs[0] = m.flusher.Start()
	}()
	go func() {
		defer wg.Done()
		errs[1] = m.bufferedSink.Start()
	}()
	wg.Wait()
	for _, err := range errs {
		if err != nil {
			return err
		}
	}
	return nil
}

func (m *Service) Close() error {
	if err := m.flusher.Close(); err != nil {
		return err
	}
	return m.bufferedSink.Close()
}

func dims(pId identifier.ProcessIdentifier) map[string]string {
	// just copy pasta from previous dims function on custom process identifier class, just want parity when porting to amzn
	ret := make(map[string]string)
	if pId.Service != "" {
		ret["Service"] = pId.Service
	}
	// Why send region if we're using cloudwatch(???)
	if pId.Stage != "" {
		ret["Stage"] = pId.Stage
	}
	if pId.Substage != "" {
		ret["Substage"] = pId.Substage
	}
	return ret
}
