package twitchcircuit

import (
	"sync"
	"time"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/log"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/cep21/circuit"
	"github.com/cep21/circuit/closers/hystrix"
	"github.com/cep21/circuit/metrics/responsetimeslo"
	"github.com/cep21/circuit/metrics/rolling"
	"github.com/cep21/circuit/metrics/statsdmetrics"
)

// Service helps using and statting circuits
type Service struct {
	Circuit                circuit.Manager
	Statsd                 statsd.SubStatter
	SLOConfig              map[string]responsetimeslo.Config
	OpenerConfig           map[string]hystrix.ConfigureOpener
	CloserConfig           map[string]hystrix.ConfigureCloser
	CircuitConfig          map[string]circuit.Config
	Log                    log.Logger
	circuitMetricCollector *statsdmetrics.ConcurrencyCollector
	Config                 *distconf.Distconf

	once sync.Once
}

func (s *Service) responseSLO(circuitName string) responsetimeslo.Config {
	if ret, exists := s.SLOConfig[circuitName]; exists {
		return ret
	}
	return responsetimeslo.Config{
		MaximumHealthyTime: s.Config.Duration("circuit."+circuitName+".slo.max_healthy", time.Millisecond*300).Get(),
	}
}

func (s *Service) hystrixOpener(circuitName string) hystrix.ConfigureOpener {
	if ret, exists := s.OpenerConfig[circuitName]; exists {
		return ret
	}
	return hystrix.ConfigureOpener{
		ErrorThresholdPercentage: s.Config.Int("circuit."+circuitName+".opener.errorThresholdPercentage", 50).Get(),
		RequestVolumeThreshold:   s.Config.Int("circuit."+circuitName+".opener.requestVolumeThreshold", 20).Get(),
		RollingDuration:          s.Config.Duration("circuit."+circuitName+".opener.rollingDuration", time.Second*10).Get(),
		NumBuckets:               int(s.Config.Int("circuit."+circuitName+".opener.numBuckets", 10).Get()),
	}
}

// Setup inits the circuit library
func (s *Service) Setup() error {
	s.once.Do(func() {
		statCommandFactory := statsdmetrics.CommandFactory{
			StatSender: s.Statsd.NewSubStatter("circuits"),
			SampleRate: float32(s.Config.Float("circuits.statsd_sample_rate", 1).Get()),
		}

		sloTracker := responsetimeslo.Factory{
			CollectorConstructors: []func(circuitName string) responsetimeslo.Collector{statCommandFactory.SLOCollector},
			ConfigConstructor:     []func(circuitName string) responsetimeslo.Config{s.responseSLO},
		}

		hconfig := hystrix.Factory{
			CreateConfigureCloser: []func(circuitName string) hystrix.ConfigureCloser{s.hystrixCloser},
			CreateConfigureOpener: []func(circuitName string) hystrix.ConfigureOpener{s.hystrixOpener},
		}

		rollingStats := rolling.StatFactory{}

		s.Circuit.DefaultCircuitProperties = append(s.Circuit.DefaultCircuitProperties,
			statCommandFactory.CommandProperties, sloTracker.CommandProperties, rollingStats.CreateConfig, s.circuitConfig, hconfig.Configure)

		s.circuitMetricCollector = statCommandFactory.ConcurrencyCollector(&s.Circuit)
	})
	return nil
}

func (s *Service) circuitConfig(circuitName string) circuit.Config {
	if ret, exists := s.CircuitConfig[circuitName]; exists {
		return ret
	}
	return circuit.Config{
		General: circuit.GeneralConfig{
			GoLostErrors: s.circuitLostErrors,
		},
		Execution: circuit.ExecutionConfig{
			Timeout:               s.Config.Duration("circuit."+circuitName+".execution.timeout", time.Second).Get(),
			MaxConcurrentRequests: s.Config.Int("circuit."+circuitName+".execution.max_concurrent", 50).Get(),
		},
	}
}

func (s *Service) circuitLostErrors(err error, panics interface{}) {
	s.Log.Log("err", err, "panic_result", panics)
}

func (s *Service) hystrixCloser(circuitName string) hystrix.ConfigureCloser {
	if ret, exists := s.CloserConfig[circuitName]; exists {
		return ret
	}
	return hystrix.ConfigureCloser{
		HalfOpenAttempts:             s.Config.Int("circuit."+circuitName+".closer.halfOpenAttempts", 1).Get(),
		RequiredConcurrentSuccessful: s.Config.Int("circuit."+circuitName+".closer.requiredSuccessful", 1).Get(),
		SleepWindow:                  s.Config.Duration("circuit."+circuitName+".closer.sleepWindow", time.Second*10).Get(),
	}
}

// Start metric collection
func (s *Service) Start() error {
	s.circuitMetricCollector.Start()
	return nil
}

// Close ends metric collection
func (s *Service) Close() error {
	return s.circuitMetricCollector.Close()
}
