package twitchparamstore

import (
	"context"
	"fmt"
	"os"
	"path/filepath"
	"sync"
	"time"

	"code.justin.tv/hygienic/distcache"

	"code.justin.tv/hygienic/errors"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/paramstoreconf"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ssm"
	"golang.org/x/sync/errgroup"
)

// Logger allows logging failures during startup
type Logger interface {
	Log(keyvals ...interface{})
}

// Service is service configuration
type Service struct {
	// The servicename
	Service string
	// Which service environment that is running
	Environment string
	// FallbackEnvironment controls another environment that can variables can be pulled from if not in the primary
	// environment
	FallbackEnvironment string
	// AWS client to pull SSM information from
	SSM *ssm.SSM
	// LocalConf allows setting values from memory
	LocalConf distconf.InMemory
	// Can be empty: defaults to config/${environment}.json
	ConfigFilename string
	// A base directory to add to ConfigFilename
	BaseDirectory    string
	SSMSessionConfig []*aws.Config
	UseLocalCache    bool

	DistConf distconf.Distconf

	// DefaultReaders are optional reads to add to distconf
	DefaultReaders []distconf.Reader

	// RefreshInterval is how frequently to refresh distconf configuration
	RefreshInterval time.Duration
	GetEnv          func(string) string

	// Optional logger during execution
	Log Logger

	cachesToStart []*distcache.DistCache
	onClose       chan struct{}
	once          sync.Once
	// If true, will also pull secrets from SSM
	AllowSSMSecrets bool

	readersToRefresh []refreshableReader
}

type refreshableReader interface {
	distconf.Reader
	distconf.Refreshable
}

func (s *Service) getEnv(key string) string {
	if s.GetEnv == nil {
		return os.Getenv(key)
	}
	return s.GetEnv(key)
}

func (s *Service) log(keyvals ...interface{}) {
	if s.Log != nil {
		s.Log.Log(keyvals...)
	}
}

func (s *Service) setupDefaults() error {
	if s.Environment == "" {
		s.Environment = s.getEnv("ENVIRONMENT")
	}
	if s.Environment == "" {
		return errors.New("unable to detect ENVIRONMENT")
	}
	if s.SSM == nil {
		awsSession, err := session.NewSession(s.SSMSessionConfig...)
		if err != nil {
			return err
		}
		s.SSM = ssm.New(awsSession)
	}
	if s.ConfigFilename == "" {
		s.ConfigFilename = fmt.Sprintf("%sconfig/%s.json", s.BaseDirectory, s.Environment)
	}
	return nil
}

func fileExists(filename string) bool {
	_, err := os.Stat(filename)
	return err == nil
}

// Setup the service for use
func (s *Service) Setup() error {
	if err := s.setupDefaults(); err != nil {
		return errors.Wrap(err, "unable to setup defaults")
	}
	commandline := distconf.CommandLine{Prefix: "--conf:"}

	var baseConfig distconf.JSONConfig
	if fileExists(s.ConfigFilename) {
		if err := baseConfig.RefreshFile(s.ConfigFilename); err != nil {
			return err
		}
	}

	envConf := distconf.Env{
		OsGetenv: s.getEnv,
	}

	fullReaders := make([]distconf.Reader, 0, len(s.DefaultReaders)+10)
	fullReaders = append(fullReaders, s.DefaultReaders...)
	fullReaders = append(fullReaders, &commandline, &baseConfig, &envConf, &s.LocalConf)

	if s.FallbackEnvironment == "" {
		rootConfig := distconf.Distconf{
			Readers: fullReaders,
		}
		s.FallbackEnvironment = rootConfig.Str("fallback_config", "").Get()
		rootConfig.Close()
	}

	serviceConfBase := paramstoreconf.ParameterStoreConfiguration{
		Service:        s.Service,
		Environment:    s.Environment,
		SSM:            s.SSM,
		AllowEncrypted: s.AllowSSMSecrets,
	}
	var serviceConf distconf.Reader
	if s.UseLocalCache {
		serviceConfCache := &distcache.DistCache{
			LocalFilename: filepath.Join(cacheDir(), "ssm_"+s.Service+"_"+s.Environment),
			Fallback:      &serviceConfBase,
			Logger:        s.Log,
		}
		serviceConf = serviceConfCache
		s.cachesToStart = append(s.cachesToStart, serviceConfCache)
	} else {
		serviceConf = &serviceConfBase
	}
	fullReaders = append(fullReaders, serviceConf)
	s.readersToRefresh = append(s.readersToRefresh, &serviceConfBase)
	if s.FallbackEnvironment != "" {
		fallbackServiceConfBase := &paramstoreconf.ParameterStoreConfiguration{
			Service:        s.Service,
			Environment:    s.FallbackEnvironment,
			SSM:            s.SSM,
			AllowEncrypted: s.AllowSSMSecrets,
		}
		var fallbackServiceConf distconf.Reader
		if s.UseLocalCache {
			fallbackCacheConf := &distcache.DistCache{
				LocalFilename: filepath.Join(cacheDir(), "ssm_"+s.Service+"_"+s.FallbackEnvironment),
				Fallback:      fallbackServiceConfBase,
				Logger:        s.Log,
			}
			fallbackServiceConf = fallbackCacheConf
			s.cachesToStart = append(s.cachesToStart, fallbackCacheConf)
		} else {
			fallbackServiceConf = fallbackServiceConfBase
		}
		fullReaders = append(fullReaders, fallbackServiceConf)
		s.readersToRefresh = append(s.readersToRefresh, fallbackServiceConfBase)
	}

	if err := s.verifyReaders(fullReaders); err != nil {
		return errors.Wrap(err, "unable to verify readers")
	}

	s.DistConf = distconf.Distconf{
		Readers: fullReaders,
	}
	return nil
}

func (s *Service) verifyReaders(fullReaders []distconf.Reader) error {
	eg, _ := errgroup.WithContext(context.Background())
	for _, reader := range fullReaders {
		reader := reader
		eg.Go(func() error {
			// All readers should be able to read without error a key of name `setuptest`
			_, err := reader.Get("setuptest")
			return errors.Wrapf(err, "unable to fetch from reader %s", reader)
		})
	}
	return eg.Wait()
}

func (s *Service) init() {
	s.once.Do(func() {
		s.onClose = make(chan struct{})
	})
}

func cacheDir() string {
	ret, err := os.UserCacheDir()
	if err == nil {
		return ret
	}
	return os.TempDir()
}

// Start periodically refreshes distconf configuration from source of truth
func (s *Service) Start() error {
	s.init()
	for _, c := range s.cachesToStart {
		go c.Start()
	}
	for {
		select {
		case <-s.onClose:
			return nil
		case <-time.After(s.refreshInterval()):
			for _, r := range s.readersToRefresh {
				r.Refresh()
			}
		}
	}
}

// Close stops periodic refreshes
func (s *Service) Close() error {
	s.init()
	s.DistConf.Close()
	close(s.onClose)
	return nil
}

func (s *Service) refreshInterval() time.Duration {
	if s.RefreshInterval == 0 {
		return time.Minute * 5
	}
	return s.RefreshInterval
}
