package twitchrolllog

import (
	"context"
	"encoding/json"
	"expvar"
	"os"
	"sync/atomic"
	"time"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/log"
	"code.justin.tv/hygienic/log/fmtlogger"
	"code.justin.tv/hygienic/rollbar"
	"code.justin.tv/hygienic/rolllog"
)

var codeVersion string

// Config configures all the rollbar logging information
type Config struct {
	rolllog.Config
	DebugEnabled  *distconf.Bool
	StdoutEnabled *distconf.Bool
}

// MarshalJSON config information without the access token
func (c Config) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"send_timeout":   c.Config.SendTimeout.Get(),
		"debug_enabled":  c.DebugEnabled.Get(),
		"stdout_enabled": c.StdoutEnabled.Get(),
	})
}

// Load rollbar info from distconf and a secrets config (for access token)
func (c *Config) Load(dconf *distconf.Distconf, secrets *distconf.Distconf) error {
	c.Config.Load(dconf)
	c.Config.LoadSecrets(secrets)
	return nil
}

// Service configures a rollbar logger
type Service struct {
	// Config must be registered before starting the service
	Config   Config
	Log      log.Logger
	DebugLog log.Logger

	CodeVersion string
	Environment string

	rollbarClient *rollbar.Client

	chanLogger    *log.ChannelLogger
	rollbarLogger log.MultiLogger
	rollbarErrors int64
	closeSignal   chan struct{}
}

func (s *Service) codeVersion() string {
	if s.CodeVersion == "" {
		if codeVersion == "" {
			return "UNKNOWN"
		}
		return codeVersion
	}
	return s.CodeVersion
}

func (s *Service) environment() string {
	if s.Environment == "" {
		return os.Getenv("ENVIRONMENT")
	}
	return s.Environment
}

// Setup ensures that rollbar logging works
func (s *Service) Setup() error {
	if s.Log != nil {
		// Logging already set.  Don't bother setting it up
		return nil
	}

	s.closeSignal = make(chan struct{})

	clientDefaults := rollbar.DataOptionals{}
	clientDefaults.MergeFrom(&rollbar.DefaultConfiguration)
	clientDefaults.CodeVersion = s.codeVersion()

	s.rollbarClient = &rollbar.Client{
		AccessToken:     s.Config.AccessToken.Get(),
		Environment:     s.environment(),
		MessageDefaults: &clientDefaults,
	}

	rollbarLogger := &rolllog.Rolllog{
		OnErr: func(err error) {
			atomic.AddInt64(&s.rollbarErrors, 1)
		},
		Client:          s.rollbarClient,
		DefaultErrLevel: rollbar.Error,
		DefaultMsgLevel: rollbar.Info,
	}
	s.Config.Monitor(rollbarLogger)
	multiLogger := log.MultiLogger([]log.Logger{
		rollbarLogger,
	})
	if s.Config.StdoutEnabled.Get() {
		stdoutLog := fmtlogger.NewLogfmtLogger(os.Stdout, log.Discard)
		multiLogger = log.MultiLogger([]log.Logger{
			rollbarLogger, stdoutLog,
		})
	}
	s.chanLogger = &log.ChannelLogger{
		Out:    make(chan []interface{}, 256),
		OnFull: log.Discard,
	}

	s.rollbarLogger = multiLogger

	normalLogLevel := &StackLogger{
		RootLogger: s.chanLogger,
	}

	debugLog := &log.Gate{
		DisabledFlag: 1,
		Logger:       log.NewContext(normalLogLevel).With("level", rollbar.Debug),
	}
	s.Log = normalLogLevel
	s.DebugLog = debugLog
	watch2 := func() {
		if s.Config.DebugEnabled.Get() {
			debugLog.Enable()
		} else {
			debugLog.Disable()
		}
	}
	watch2()
	s.Config.DebugEnabled.Watch(watch2)
	return nil
}

// MarshalJSON exposes some debug information
func (s *Service) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"rollbar":        expvarToVal(s.rollbarClient.Var()),
		"rollbar_errors": atomic.LoadInt64(&s.rollbarErrors),
		"config":         s.Config,
	})
}

func expvarToVal(in expvar.Var) interface{} {
	type iv interface {
		Value() interface{}
	}
	if rawVal, ok := in.(iv); ok {
		return rawVal.Value()
	}
	return nil
}

var _ json.Marshaler = &Service{}

// Start drains into rollbar
func (s *Service) Start() error {
	log.DrainChannel(s.rollbarLogger, s.chanLogger.Out, s.closeSignal)
	return nil
}

// Close stops draining
func (s *Service) Close() error {
	close(s.closeSignal)
	return nil
}

// Verify rollbar client logging
func (s *Service) Verify() error {
	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()
	_, err := s.rollbarClient.Message(ctx, rollbar.Debug, "verifying rollbar logger")
	return errors.Wrap(err, "unable to verify rollbar logger")
}
