package twitchrolllog

import (
	"fmt"
	"reflect"

	"code.justin.tv/hygienic/errors"
	"code.justin.tv/hygienic/log"
)

// StackLogger checks for any err keys and wraps them if needed so that we get a pretty stack trace in the finished log
// because the stack trace is eaten when we pass the log onto a channel
type StackLogger struct {
	RootLogger log.Logger
}

// Log keyvals attaching a stack trace to errors
func (s *StackLogger) Log(keyvals ...interface{}) {
	for idx := 0; idx+1 < len(keyvals); idx += 2 {
		if mapKey(keyvals[idx]) == "err" {
			if err, ok := keyvals[idx+1].(error); ok {
				keyvals[idx+1] = wrapErrIfNeeded(err)
			}
		}
	}
	s.RootLogger.Log(keyvals...)
}

func mapKey(k interface{}) (s string) {
	defer func() {
		if panicVal := recover(); panicVal != nil {
			s = nilCheck(k, panicVal, "NULL").(string)
		}
	}()
	switch x := k.(type) {
	case string:
		return x
	case fmt.Stringer:
		return x.String()
	default:
		return fmt.Sprint(x)
	}
}

func nilCheck(ptr, panicVal interface{}, onError interface{}) interface{} {
	if vl := reflect.ValueOf(ptr); vl.Kind() == reflect.Ptr && vl.IsNil() {
		return onError
	}
	panic(panicVal)
}

func wrapErrIfNeeded(err error) error {
	type tracedError1 interface {
		StackTrace() []uintptr
	}
	type tracedError2 interface {
		Stack() []uintptr
	}
	if _, ok := err.(tracedError1); ok {
		return err
	}
	if _, ok := err.(tracedError2); ok {
		return err
	}
	return errors.Wrap(err, "")
}
