package workerpool

import (
	"fmt"
	"sync"
	"time"
)

// A Future is a job sent to the Pool that can either be executed eventually, or executed explicitly with Done()
type Future struct {
	// The function to execute
	callback Runnable
	// if not nil, a captured panic that is rethrown during result
	panicResult interface{}
	// The result of the calculation
	result interface{}
	// closed when a result is calculated
	done chan struct{}
	// Ensures we calculate a result once
	once sync.Once
	// how long the execute step took
	executeTime time.Duration
}

// NewFuture returns a new initialized future
func NewFuture(callback Runnable) *Future {
	return &Future{
		callback: callback,
		done:     make(chan struct{}),
	}
}

// String returns a string representation of the future, telling if it is done.
func (f *Future) String() string {
	if f == nil {
		return "<nil future>"
	}
	var isDone bool
	select {
	case <-f.done:
		isDone = true
	default:
		isDone = false
	}
	return fmt.Sprintf("<future done=%t>", isDone)
}

// Result executes the callback and close the Done channel.  It is safe to call Result multiple times.  If the callback
// panics, so will result.
func (f *Future) Result() interface{} {
	f.execute()
	if f.panicResult != nil {
		panic(f.panicResult)
	}
	return f.result
}

// Done returns a channel that is closed when Execute is finished.  It mirrors the context.Context.Done API.
func (f *Future) Done() <-chan struct{} {
	if f == nil {
		return nil
	}
	return f.done
}

func (f *Future) execute() {
	f.once.Do(func() {
		startTime := time.Now()
		defer func() {
			f.executeTime = time.Since(startTime)
			f.panicResult = recover()
			close(f.done)
		}()
		f.result = f.callback()
	})
}
