package workerpool_test

import (
	"context"
	"sync/atomic"
	"testing"
	"time"

	"code.justin.tv/hygienic/workerpool"
	. "github.com/smartystreets/goconvey/convey"
)

func happyRunnable() interface{} {
	return "hi"
}

func panicRunnable() interface{} {
	panic("panic!")
}

type atomicInt struct {
	count int64
}

func (a *atomicInt) incr() interface{} {
	return atomic.AddInt64(&a.count, 1)
}

func TestPool(t *testing.T) {
	Convey("a pool", t, func() {
		ctx := context.Background()
		p := workerpool.Pool{
			FuturesChannel: make(chan *workerpool.Future, 2),
		}
		Reset(func() {
			p.Close()
		})
		var a atomicInt
		f := p.Offer(happyRunnable)
		So(f.String(), ShouldContainSubstring, "<future done=false>")
		So(p.String(), ShouldContainSubstring, "<pool ")
		So(f.Result().(string), ShouldEqual, "hi")
		So(p.Put(ctx, a.incr), ShouldNotBeNil)
		// Without any workers, it shouldn't ever happen
		time.Sleep(time.Millisecond * 100)
		So(atomic.LoadInt64(&a.count), ShouldEqual, 0)
		Convey("Should Do if full", func() {
			// Now it is full, should block on puts
			ctx2, cancel := context.WithTimeout(ctx, time.Millisecond*20)
			defer cancel()
			So(p.Put(ctx2, happyRunnable), ShouldBeNil)
			f := workerpool.OfferOrDo(&p, happyRunnable)
			So(f, ShouldNotBeNil)
			So(f.Result().(string), ShouldEqual, "hi")
		})
		Convey("With workers", func() {
			p.SetNumberOfWorkers(10)
			Convey("Should scale down and work", func() {
				p.SetNumberOfWorkers(1)
				f := p.Put(context.Background(), happyRunnable)
				<-f.Done()
				So(f.Result().(string), ShouldEqual, "hi")
			})
			Convey("added incr should happen", func() {
				// Should eventually execute the count
				j := 0
				for atomic.LoadInt64(&a.count) != 1 && j < 10000 {
					time.Sleep(time.Millisecond)
					j++
				}
				So(atomic.LoadInt64(&a.count), ShouldEqual, 1)
				Convey("should show stats", func() {
					So(p.Var().String(), ShouldContainSubstring, "2")
					So(p.Stats().TotalItems, ShouldEqual, 2)
				})
			})
			Convey("panics should happen", func() {
				f := p.Put(ctx, panicRunnable)
				So(f, ShouldNotBeNil)
				<-f.Done()
				So(func() {
					f.Result()
				}, ShouldPanic)
			})
		})
	})
}
