const fs = require("fs");
const path = require("path");

const localesInfo = require("../../src/locales.json");
const { distDir, getLocaleFile } = require("./utils");

/**
 * For these locales we will not abbreviate certain sizes of numbers.
 * @see https://jira.twitch.com/browse/BROWSER-4905
 * @see https://jira.twitch.com/browse/BROWSER-7139
 */
const dropAbbreviations = {
  "ko-kr": new Set(["1000"]),
  "es-es": new Set(["1000", "10000", "100000"]),
};

/**
 * These are the levels we include short rules for
 * (unlikely to display numbers larger than 10B)
 */
const numberLevels = new Set([
  "1000",
  "10000",
  "100000",
  "1000000",
  "10000000",
  "100000000",
  "1000000000",
  "10000000000",
]);

// if (!fs.existsSync("../../dist/locale-data")) {
// fs.mkdirSync("../../dist/locale-data");
if (!fs.existsSync(path.join(distDir, "locale-data/number-short"))) {
  fs.mkdirSync(path.join(distDir, "locale-data/number-short"), {
    recursive: true,
  });
}
// }

function getNumberModule(locale) {
  return `cldr-numbers-modern/main/${locale}/numbers.json`;
}

Object.keys(localesInfo.cldrLocales).map((localeKey) => {
  const locale = localesInfo.cldrLocales[localeKey];
  let numberShortData;
  try {
    const cldrNumbers = getLocaleFile(locale, getNumberModule);
    numberShortData =
      cldrNumbers.numbers["decimalFormats-numberSystem-latn"].short
        .decimalFormat;
  } catch (e) {
    console.error(e);
    console.error("Invalid key/locale:", { localeKey, locale });
    throw new Error("Invalid key/locale, aborting build");
  }

  const filteredData = {};
  Object.keys(numberShortData).forEach((type) => {
    const numStr = type.slice(0, type.indexOf("-")); // converting "1000-count-one" to 1000

    if (numberLevels.has(numStr)) {
      if (
        dropAbbreviations[localeKey] &&
        dropAbbreviations[localeKey].has(numStr)
      ) {
        // no abbreviation, just format the entire number
        filteredData[type] = "0";
      } else {
        // Replace escaped periods with real periods
        filteredData[type] = numberShortData[type].replace(/'\.'/g, ".");
      }
    }
  });

  fs.writeFileSync(
    path.join(distDir, `locale-data/number-short/${localeKey}.json`),
    JSON.stringify(filteredData, null, 2)
  );
});
