/* jshint node: true */

'use strict';

/**
 * Copyright 2015, Yahoo! Inc.
 * Copyrights licensed under the New BSD License. See the accompanying LICENSE file for terms.
 */

let stringify = require('json-stable-stringify');
let WatchedDir = require('broccoli-source').WatchedDir;
let existsSync = require('exists-sync');
let path = require('path');
let utils = require('./lib/utils');
let mergeTrees = require('broccoli-merge-trees');
let TranslationReducer = require('./lib/broccoli/translation-reducer');
let funnel = require('broccoli-funnel');

let convertJsonToJs = require('i18n-utils-twitch').convertJsonToJs;

module.exports = {
  name: 'ember-intl-i18n-twitch',
  opts: null,
  isLocalizationFramework: true,

  included() {
    this._super.included.apply(this, arguments);

    let app = this.app = this._findParentApp();
    this.opts = this.intlConfig(app.env);

    let inputPath = this.opts.inputPath;
    this.hasTranslationDir = existsSync(path.join(app.project.root, inputPath));

    this.translationTree = new WatchedDir(inputPath);

    let nodes = [this.translationTree];
    if (this.opts.buildTranslationsPath && existsSync(path.join(app.project.root, this.opts.buildTranslationsPath))) {
      this.log('Using Build-Translations folder', true);
      let buildTranslationsNode = funnel(new WatchedDir(this.opts.buildTranslationsPath), {
        exclude: [this.opts.baseLocaleFileNameInZip] // Disregard en-us.yaml from build folder
      });
      nodes.push(buildTranslationsNode);
    }

    if (this.opts.phoenixBuildTranslationsPath && existsSync(path.join(app.project.root, this.opts.phoenixBuildTranslationsPath))) {
      this.log('Using Phoenix-Build-Translations folder', true);
      let phoenixBuildTranslationsNode = funnel(new WatchedDir(this.opts.phoenixBuildTranslationsPath), {
        getDestinationPath: (relativePath) => {
          return 'phoenix/' + relativePath;
        },
        exclude: [this.opts.baseLocaleFileNameInZip] // Disregard en-us.yaml from build folder
      });
      nodes.push(phoenixBuildTranslationsNode);
    }

    this.translationTree = mergeTrees(nodes, {overwrite: true});
  },

  _findParentApp() {
    if (typeof this._findHost === 'function') {
      return this._findHost();
    }

    let current = this;
    let app;

    // Keep iterating upward until we don't have a grandparent.
    // Has to do this grandparent check because at some point we hit the project.
    do {
      app = current.app || app;
    } while (current.parent.parent && (current = current.parent));

    return app;
  },

  outputPaths() {
    let assetPath = 'assets/intl';
    let appOptions = this.app.options;

    if (appOptions.app && appOptions.app.intl) {
      assetPath = appOptions.app.intl;
    }

    return assetPath;
  },

  treeForApp(tree) {
    let trees = (tree == undefined) ? [] : [tree];

    if (this.hasTranslationDir && this.opts.includeBaseLocaleInApp) { 
      let baseLocaleArr = [this.opts.baseLocale];  
      trees.push(this.reduceTranslations({ 
        filename: function filename(key) { 
          return key + '.js'; 
        }, 
        wrapEntry: function wrapEntry(obj) { 
          return 'export default ' + stringify(obj) + ';';
        },
        locales: baseLocaleArr ,
        webClientOnly: true
      }));
    }

    return mergeTrees(trees, { overwrite: true });
  },

  treeForPublic() {
    let publicTree = this._super.treeForPublic.apply(this, arguments);
    let trees = [];

    if (publicTree) {
      trees.push(publicTree);
    }

    if (this.hasTranslationDir) {
      trees.push(this.reduceTranslations({
        filename: function filename(key) {
          return key + '.js';
        },
        wrapEntry: convertJsonToJs,
        webClientOnly: false
      }));
    }

    return mergeTrees(trees, { overwrite: true });
  },

  log(msg, forceLog) {
    if ((typeof forceLog !== 'undefined' && !forceLog) || this.app.options && this.app.options.intl && this.app.options.intl.silent) {
      return;
    }

    this.ui.writeLine('[ember-intl-i18n-twitch] ' + msg);
  },

  readConfig(environment) {
    let project = this.app.project;

    // NOTE: For ember-cli >= 2.6.0-beta.3, project.configPath() returns absolute path
    // while older ember-cli versions return path relative to project root
    let configPath = path.dirname(project.configPath());
    let config = path.join(configPath, 'ember-intl-i18n-twitch.js');

    if (!path.isAbsolute(config)) {
      config = path.join(project.root, config);
    }

    if (existsSync(config)) {
      return require(config)(environment);
    }

    return {};
  },

  intlConfig(environment) {
    let deprecatedConfig = this.app.project.config(environment)['intl'];
    let addonConfig = Object.assign(this.readConfig(environment), deprecatedConfig || {});

    if (deprecatedConfig) {
      this.log('DEPRECATION: intl configuration should be moved into config/ember-intl.js');
      this.log('Run `ember g ember-intl-config` to create a default config');
    }

    if (addonConfig.defaultLocale) {
      this.log('DEPRECATION: defaultLocale is deprecated in favor of baseLocale');
      this.log('Please update config/ember-intl.js or config/environment.js');
      addonConfig.baseLocale = addonConfig.defaultLocale;
    }

    addonConfig = Object.assign({
      locales: null,
      baseLocale: null,
      inputPath: './i18n/translations',
      outputPath: 'translations',
      includeBaseLocaleInApp: false
    }, addonConfig);

    if (addonConfig.locales) {
      addonConfig.locales = utils.castArray(addonConfig.locales).filter(function(locale) {
        return typeof locale === 'string';
      }).map(function(locale) {
        return locale.toLocaleLowerCase();
      });
    }

    return addonConfig;
  },

  mergeTranslationTrees(projectTranslations, addonTranslations) {
    let trees = [];

    trees.push(projectTranslations);

    if (addonTranslations && addonTranslations.length) {
      trees = trees.concat(addonTranslations);
    }

    return mergeTrees(trees);
  },

  reduceTranslations(opts) {
    if (!opts) { opts = {}; }
    let addon = this;

    return new TranslationReducer([this.translationTree], Object.assign({}, this.opts, opts, {
      log() {
        return addon.log.apply(addon, arguments);
      }
    }));
  }
};