var asyncUtil        = require('async'),
    path             = require('path'),
    Q                = require('q'),
    _                = require('lodash'),
    request          = require('request'),
    jszip            = require('jszip'),
    currPkgVersion   = require('./../package.json').version,
    semver           = require('semver');

var ASYNC_OPS_LIMIT = 7;

function formatFileNameHelper(fileName, fileType) {
  return fileName + '.' + fileType;
}

function handleRequestError(error, response, body, callback) {
  if (error && !body) {
    return callback(error);
  } else {
    var errMsg = "\nResponse Code: " + response.statusCode;
    errMsg += "\nResponse Body: " + (body || '').toString();
    return callback(new Error(errMsg));
  }
}

module.exports = {
  formatFileName: formatFileNameHelper,
  setI18nType: function (grunt, i18nConfig) {

    var i18nTypes = require('./i18n-types/index.js');
    if (!i18nTypes.hasOwnProperty(i18nConfig.i18nType)) {
      grunt.fail.fatal('Invalid I18n Type: ' + i18nConfig.i18nType);
    }

    var I18nType = i18nTypes[i18nConfig.i18nType];
    grunt.option('i18nType', new I18nType(grunt, i18nConfig));
    return grunt.option('i18nType');
  },

  checkVersion: function(grunt, options, callback) {
    grunt.verbose.writeln('Checking version against server');

    request.get({
      url: options.i18n.serviceUrl + '/version'
    }, function (error, response, body) {
      if (!error && response.statusCode === 200) {
        var jsBody = JSON.parse(body);

        grunt.verbose.writeln('Minimum Version: ' + jsBody.minScriptVersion + '. Actual: ' + currPkgVersion);
        if (semver.gt(jsBody.minScriptVersion, currPkgVersion)) {
          return callback(grunt.util.error('Please update to latest grunt-smartling-sdk-twitch versions. (Probably rebase/pull in master)'));
        }
        return callback();
      } else {
        handleRequestError(error, response, body, callback);
      }
    });
  },

  confirmFileExists: function (sdk, grunt, options, fileUri, callback) {
    sdk.filesList(fileUri, options.i18n.fileType)
        .then(function (filesList) {

          if (filesList.items) {
            var hasExpectedFileUri = filesList.items.some(function (elem) {
              return elem.fileUri === fileUri;
            });
            if (hasExpectedFileUri) {
              return callback(null, true);
            }
          }

          return callback(null, false);
        })
        .fail(function (error) {
          if (!_.isError(error)) {
            error = grunt.util.error('Error Reaching to Smartling: ' + JSON.stringify(error, null, 2));
          }

          return callback(error);
        });
  },

  clearBuildCache: function(options, callback) {
    request.del({
      url: options.i18n.serviceUrl + '/api/build/v1/cache',
      qs: {
        projectId: options.i18n.projectId
      }
    }, function(error, response, body) {
      if (!error && response.statusCode === 200) {
        return callback();
      } else {
        handleRequestError(error, response, body, callback);
      }
    });
  },

  diffFromSmartlingSource: function(sdk, grunt, i18nConfig, callback) {
    var smartlingSourceFileUri = formatFileNameHelper(i18nConfig.baseLocale, i18nConfig.fileType);
    var i18nType = grunt.option('i18nType');

    sdk.getOriginal(smartlingSourceFileUri)
        .then(function(contents) {
          var newStrings = i18nType.getNewStrings({smartlingSourceStr: contents});
          return callback(null, newStrings);
        })
        .fail(function (error) {
          if (!_.isError(error)) {
            error = grunt.util.error('Unknown Error Getting Original File from Smartling: ' + JSON.stringify(error));
          }

          return callback(error);
        });
  },

  getSmartlingCallbackUrl: function getSmartlingCallbackUrl(options, ticket, callback) {
    request.post({
      url: options.i18n.serviceUrl + '/api/build/v1/smartling-callback-url',
      qs: {
        ticket: ticket,
        projectId: options.i18n.projectId
      }
    }, function(error, response, body) {
      if (!error && response.statusCode === 200) {
        return callback(null, body);
      } else {
        handleRequestError(error, response, body, callback);
      }
    });
  },

  getBuildTranslations: function(sdk, grunt, options, commandLineOptions, callback) {
    grunt.log.writeln('Attempting to fetch build translations');
    var i18nType = grunt.option('i18nType');
    var sourceFileUri = formatFileNameHelper(options.i18n.baseLocale, options.i18n.fileType);
    var sourceFileContents = i18nType.convertToSmartlingFormat({fileStr: i18nType.getSourceFileStr()});

    if (!process.env.CI) {
      grunt.verbose.subhead('Smartling File (Disabled printing for now)');
      // grunt.verbose.writeln(sourceFileContents);
    }

    var queryParams = _.assign({
      fileUri: sourceFileUri,
      projectId: options.i18n.projectId,
      locales: options.i18n.locales.join(',')
    }, commandLineOptions);

    var req = request.post({
      url: options.i18n.serviceUrl + '/api/build/v1/get-translations',
      qs: queryParams,
      encoding: null
    }, function(error, response, body) {
      if (!error && response.statusCode === 200) {
        jszip.loadAsync(body).then(function (zip) {
          return callback(null, zip);
        });
      } else {
        handleRequestError(error, response, body, function (error) {
          // This will always return just an error, putting this here to catch errors
          grunt.log.error('Get Build Translations Failed: ', error);
          return callback(error);
        });
      }
    });

    var form = req.form();
    form.append('file', sourceFileContents, {
      filename: sourceFileUri,
      contentType: 'text/plain'
    });
  },

  unzipTranslations: function (grunt, zipFile, options, callback) {
    var i18nType = grunt.option('i18nType');

    var promises = [];
    zipFile.forEach(function (relativePath, file) {
      var promise = file.async('string').then(function (content) {
        var localeFileName = path.basename(relativePath);
        var locale = path.basename(localeFileName, '.' + options.i18n.baseLocale);

        var translationFilePath = path.join(process.cwd(), i18nType.getBuildFilePath({locale: locale, fileName: localeFileName}));
        var fileContents = i18nType.convertFromSmartlingFormat({fileStr: content});
        grunt.file.write(translationFilePath, fileContents);
      });

      promises.push(promise);
    });

    Q.allSettled(promises)
        .then(function (results) {
          var i = 0;
          for (i; i < results.length; i++) {
            var result = results[i];
            if (result.state === 'rejected') {
              return callback(grunt.util.error(result.reason));
            }
          }
          return callback();
        });
  },

  addSubmitJiraComment: function addSubmitJiraComment(grunt, options, formData, callback) {
    var req = request.post({
      url: options.i18n.serviceUrl + '/v1/jira/submit'
    }, function(error, response, body) {
      if (!error && response.statusCode === 200) {
        return callback();
      } else {
        handleRequestError(error, response, body, callback);
      }
    });

    var form = req.form();
    form.append('ticket', formData.ticket);
    form.append('projectId', formData.projectId);
    form.append('fileUri', formData.fileUri);
    form.append('file', formData.submittedStrings, {
      filename: 'submittedStrings',
      contentType: 'text/plain'
    });
  },

  postStatusToI18nService: function (grunt, options, qs, callback) {
    var mergedQs = _.assign({}, qs, {packageVersion: currPkgVersion});
    request.post({
      url: options.i18n.serviceUrl + '/api/build/v1/script-error',
      qs: mergedQs
    }, function(postScriptIssueError, response, body) {
      if (!postScriptIssueError && response.statusCode === 200) {
        grunt.verbose.writeln('Posted build error correctly');
      } else if (postScriptIssueError) {
        grunt.verbose.writeln('Could not post error correctly. Body: ' + body + ' statusCode: ' + response.statusCode);
      }

      return callback();
    });
  },

  getTranslations: function(sdk, grunt, options, callback) {
    var i18nType = grunt.option('i18nType');
    var sourceFileUri = formatFileNameHelper(options.i18n.baseLocale, options.i18n.fileType);
    var sourceFileContents = i18nType.convertToSmartlingFormat({fileStr: i18nType.getSourceFileStr()});

    grunt.verbose.subhead('To Submit Smartling File:');
    grunt.verbose.writeln('Skipping for now');
    //grunt.verbose.writeln(sourceFileContents);

    var locales = options.i18n.locales;
    if (!options._copyBaseTranslation) {
      // Removing baseLocale if it is NOT meant to be moved
      locales = locales.filter(function (locale) {
        return locale !== options.i18n.baseLocale;
      });
    }

    asyncUtil.eachLimit(locales, ASYNC_OPS_LIMIT, function (locale, fileCallback) {
      grunt.verbose.or.writeln('Processing Locale: ', locale);
      sdk.getTranslations(sourceFileUri, sourceFileContents, locale, options.operation)
          .then(function (fileContents) {

            var filePath = i18nType.getFilePath({
              locale: locale,
              fileName: formatFileNameHelper(locale, options.i18n.fileType)
            });

            var transformedFileStr = i18nType.convertFromSmartlingFormat({fileStr: fileContents});
            grunt.file.write(
                filePath,
                transformedFileStr
            );

            return fileCallback();
          })
          .fail(function (error) {
            if (!_.isError(error)) {
              error = grunt.util.error(JSON.stringify(error, null, 2));
            }

            return fileCallback(error);
          });

    }, callback);
  },

  uploadSource: function (sdk, grunt, options, callback) {
    var i18nType = grunt.option('i18nType');
    var fileUri = formatFileNameHelper(options.i18n.baseLocale, options.i18n.fileType);
    var sourceFileStr = i18nType.getSourceFileStr();
    var smartlingFileStr = i18nType.convertToSmartlingFormat({fileStr: sourceFileStr});

    asyncUtil.waterfall([
          function checkStatus(callback) {
            sdk.getOriginal(fileUri)
                .then(function(contents) {
                  var shouldUpload = (contents !== smartlingFileStr);
                  return callback(null, shouldUpload);
                })
                .fail(function (error) {
                  if (!_.isError(error)) {
                    error = grunt.util.error('Unknown Error Getting Original File from Smartling: ' + JSON.stringify(error));
                  }

                  if (error.message.indexOf('could not be found') >= 0) {
                    return callback(null, true);
                  }

                  grunt.log.error('Error Submitting to Smartling', error);
                  return callback(error);
                });
          },
          function doUpload(shouldUpload, callback) {
            grunt.verbose.writeln('Will update base locale: ', shouldUpload);

            if (!shouldUpload) {
              return callback();
            }

            sdk.uploadContents(smartlingFileStr, fileUri, options.i18n.fileType, {})
                .then(function (fileInfo) {
                  if (fileInfo && grunt.option('verbose')) {
                    grunt.log.writeln(JSON.stringify(fileInfo, null, 2));
                  }

                  grunt.verbose.subhead('Uploaded');

                  return callback();
                })
                .fail(function (error){
                  grunt.log.error('Error Submitting to Smartling', error);
                  return callback(error);
                });
          }
        ],
        callback);
  }
};