var utils = require('i18n-utils-twitch').utils,
    yamlConvert = require('i18n-utils-twitch').yamlConvert,
    yamlIcu = require('i18n-utils-twitch').yamlIcu,
    path = require('path'),
    _ = require('lodash'),
    readlineSync  = require('readline-sync'),
    chalk = require('chalk');

function YamlICUType(grunt, i18nOptions) {
  this.grunt = grunt;
  this.i18nOptions = i18nOptions;
}

YamlICUType.prototype.constructor = YamlICUType;

/**
 * @returns {string}
 */
YamlICUType.prototype.getSourceFileStr = function () {
  return this.grunt.file.read(this.i18nOptions.sourcePath);
};

YamlICUType.prototype.convertFileStrToObject = function (fileStr) {
  return utils.loadYaml(fileStr);
};

YamlICUType.prototype.convertObjectToFileStr = function (fileStr) {
  return utils.dumpYaml(fileStr);
};

YamlICUType.prototype.removeNoTranslate = function (fileObj) {
  return utils.removeNoTranslate(fileObj);
};

/**
 * @param {string} options.smartlingSourceStr
 * @returns {string|null}
 */
YamlICUType.prototype.getNewStrings = function (options) {
  var sourceYamlObj = utils.loadYaml(this.getSourceFileStr());
  var smartlingYamlObj = yamlConvert.smartlingToPropSchema(utils.loadYaml(options.smartlingSourceStr));
  // Remember this gets keys in first arg that are NOT in second arg
  var newStringsObj = yamlIcu.newTranslations(sourceYamlObj, smartlingYamlObj);
  if (Object.keys(newStringsObj).length > 0) {
    return utils.dumpYaml(newStringsObj);
  } else {
    return null;
  }
};

/**
 * Converts either an object or file (pass in path) to Smartling File Format
 * @param {string} options.fileStr
 * @returns {string}
 */
YamlICUType.prototype.convertToSmartlingFormat = function (options) {
  var yamlObj = utils.loadYaml(options.fileStr);
  var placeholderHeader = ['# smartling.plurals_detection = on', '# smartling.placeholder_format_custom = \\{.+?\\}', '---',''].join('\n');

  return placeholderHeader + yamlIcu.toSmartlingFormat(yamlObj);
};

/**
 * @param {string} options.fileStr
 * @returns {*}
 */
YamlICUType.prototype.convertFromSmartlingFormat = function convertFromSmartlingFormat(options) {
  var yamlObj = yamlIcu.fromSmartlingFormat(utils.loadYaml(options.fileStr));
  return utils.dumpYaml(yamlObj);
};

/**
 * @param options
 * @param {string} options.fileStr
 */
YamlICUType.prototype.preSubmitCheck = function (options) {
  var sourceObj = utils.loadYaml(this.getSourceFileStr());
  var newStringsObj = utils.loadYaml(options.fileStr);

  if (this._checkCommonTranslations(sourceObj, newStringsObj)) {
    var msg = chalk.bold('Duplicate values were detected, do you want to fix them? [y/N] ');
    var confirmIgnoreDuplicates = this.grunt.option('ignoreDuplicates') || readlineSync.question(msg);
    if (confirmIgnoreDuplicates.toLowerCase() === 'y') {
      throw new Error(this.grunt.util.error('Exiting, please fix and run script again. Nothing submitted'));
    }
  }
};

/**
 * @param options
 * @param {string} options.fileName
 * @param {string} options.locale
 * @returns {*}
 */
YamlICUType.prototype.getFilePath = function getFilePath(options) {
  var defaultTranslationsFolder = this.i18nOptions.buildTranslationsDirectory || this.i18nOptions.translationsDirectory;
  return path.join(defaultTranslationsFolder, options.fileName);
};

/**
 * @param options
 * @param {string} options.fileName
 * @param {string} options.locale
 * @returns {*}
 */
YamlICUType.prototype.getBuildFilePath = function getBuildFilePath(options) {
  return path.join(this.i18nOptions.buildTranslationsDirectory, options.fileName);
};


/**
 * Should throw an error
 * @returns {boolean}
 */
YamlICUType.prototype.validateTranslations = function validateTranslations() {
  return true;
};

/**
 * Prints out warning and returns true if there are common translations
 * @param origSourceObj
 * @param newStringsObj
 * @returns {boolean}
 */
YamlICUType.prototype._checkCommonTranslations = function (origSourceObj, newStringsObj) {
  // newStringsObj are also in origSourceObj, so need to remove them
  var sourceObj = _.cloneDeep(origSourceObj);
  _.keys(newStringsObj).forEach(function(key) {
    delete sourceObj[key];
  });

  var sourceObjKeyValueString = yamlConvert.extractPhrase(sourceObj);
  var newStringsObjKeyValueString = yamlConvert.extractPhrase(newStringsObj);

  var inverted = _.invertBy(sourceObjKeyValueString);

  var hasDups = false;
  var self = this;
  _.keys(newStringsObjKeyValueString).forEach(function(key) {
    var value = newStringsObjKeyValueString[key];
    if (value in inverted) {
      self.grunt.log.subhead('This new translation is similar to others, please check if the keys can be reused:');
      self.grunt.log.writeln('Key: ' + key);
      self.grunt.log.writeln('Translation: ' + newStringsObjKeyValueString[key]);
      self.grunt.log.writeln(JSON.stringify(inverted[value]));
      hasDups = hasDups || true;
    }
  });

  return hasDups;
};

module.exports = YamlICUType;
